      ******************************************************************
      *
      *  Copyright (C) Micro Focus 2010-2013.  All rights reserved.
      *
      *  This sample code is supplied for demonstration purposes only
      *  on an "as is" basis and is for use at your own risk.
      *
      ******************************************************************

      ********************************************************************************************************
      *
      *  This project forms part of a collection of language reference examples and other material which aims
      *  to provide you with an introduction to some key aspects of the .NET framework as well as introduce
      *  any new syntax to support these features.
      *
      ********************************************************************************************************
      *
      *  Title: CoreDemo 
      *
      *  Language Items: STATIC, PROPERTY, ENUM, INTERFACE,
      *                  INVOKE, CONDITION-VALUE, PRIVATE, PROTECTED, INSTANCE OF, AS,
      *                  NULL, ATTRIBUTE,
      *                  DECLARE
      *
      *
      *  About This Program:
      *  The purpose of this program is to introduce some of the standard features of the .NET framework
      *  you will need to be aware of, and any new COBOL syntax in order to support these features.


       program-id CoreDemo.

      *> Note that some types have predefined COBOL names:

      *> Predefined COBOL name        System Class        C# equivalent  Alternative COBOL
      *> =====================        ============        =============  =================
      *> binary-char                  System.SByte        sbyte          PIC S9(2) COMP-5
      *> binary-char unsigned         System.Byte         byte           PIC 9(2) COMP-5
      *> binary-short                 System.Int16        short          PIC S9(4) COMP-5
      *> binary-short unsigned        System.UInt16       ushort         PIC 9(4) COMP-5
      *> binary-long                  System.Int32        int            PIC S9(9) COMP-5
      *> binary-long unsigned         System.UInt32       uint           PIC 9(9) COMP-5
      *> binary-double                System.Int64        long           PIC S9(18) COMP-5
      *> binary-double unsigned       System.UInt64       ulong          PIC 9(18) COMP-5
      *> character                    System.Char         char
      *> float-short                  System.Single       float          COMP-1
      *> float-long                   System.Double       double         COMP-2
      *> condition-value              System.Boolean      bool
      *> decimal                      System.Decimal      decimal
      *> object                       System.Object       object
      *> string                       System.String       string

       data division.

       working-storage section.

      *> Use predefined type 'object' meaning System.Object.
       01 obj-object     object.
      *> Other types are referenced using the TYPE keyword, followed by the case-sensitive type name
       01 obj-app-domain type System.AppDomain.
      *> Since 'System' is a predefined namespace for the project, it can be omitted in type specifiers
      *> So the following is equally valid:
       01 obj-app-domain2 type AppDomain.
      *> Similarly, this actually defines an item of type System.Type
       01 obj-type       type Type.
       01 obj-string     string.

      *> Some framework classes contain nested types. In order to reference a nested type, specify
      *> the containing type and then use '+' to reference the nested type within.
       01 cls-nested-type type ListViewItem+ListViewSubItem.

      *> It is possible to define a .Net array by specifying '[]' at the end of the external name.
      *> For instance, here we define an array of System.Objects.
       01 cls-object-array type System.Object[].
      *> However it is more convenient to access .NET arrays using standard COBOL OCCURS syntax
       01 cls-object-array object occurs any.

      *> Universal or untyped objects are equivalent to referencing System.Object explicitly.
       01 an-obj         object.

       procedure division.

      ********************************************************************************************************
      *
      *  U s a g e   o f   t h e   I N S T A N C E   O F   O p e r a t o r
      *

      * The INSTANCE OF operator is used to test the type of an object reference

      *> Set a System.Object reference to the currently executing System.AppDomain
      *> This assignment is valid because System.AppDomain inherits from System.Object.
      *> In fact System.Object is the root all objects within the framework.
           set obj-object to type System.AppDomain::CurrentDomain

      *> Though we have a reference to a System.Object we can test what type it actually references
      *> using INSTANCE OF
           if obj-object is instance of type System.AppDomain
               display "obj-object of type System.Object is referencing an instance of System.AppDomain"
           end-if

      ********************************************************************************************************
      *
      *  U s a g e   o f   t h e   T Y P E   O F   O p e r a t o r
      *

      *  Each class has an associated System.Type object which can be used to get
      *  information about the class such as its fields and methods.

      *> Use GetType() method of System.Object to obtain the System.Type object for
      *> an object reference.
           set obj-type to obj-object::GetType()

      *> The TYPE OF operator can be used to obtain the System.Type for a given
      *> class name.

           evaluate obj-type

           when type of object
               display "Will not execute"

           when type of System.AppDomain
               display "obj-type is a reference to the Type for a System.AppDomain class"

           end-evaluate


      ********************************************************************************************************
      *
      *  C a s t i n g   O b j e c t   R e f e r e n c e s   U s i n g   A S
      *

      * You can cast an object reference to an instance of another type using AS.

      *> We know obj-object points to a System.AppDomain reference so we know we can
      *> cast up the class hierarchy from System.Object to System.AppDomain. The
      *> following line if uncommented would fail as the compiler cannot determine
      *> if it is a type safe operation.

      *> set obj-app-domain to obj-object

      *> However, using the AS operator you can force a cast by using the
      *> type of the object you wish to cast to.
           set obj-app-domain to obj-object as type System.AppDomain


      *> A cast is only legal if the target type (the type being cast to) derives
      *> directly from the source type, or vice versa.
      *> So the following statement would fail during compilation.
      *> A System.Type reference (obj-type) can never point to a System.AppDomain
      *> instance because the classes are unrelated in the hierarchy.

      *> set obj-app-domain to obj-type as type System.AppDomain

      *> Be careful! If the cast fails, a System.InvalidCastException will be thrown
      *> So you should either test for a valid cast first using INSTANCE OF or wrap the cast in
      *> an exception block (see exception handling example for more information).

      *> An exception will be thrown here as we are claiming obj-object references
      *> a System.AppDomain instance where in fact it's referencing a System.Type instance
      *> The exception will be caught in the catch block, in which the special register
      *> exception-object holds the caught exception reference.

           try
               set obj-object to obj-type
               set obj-app-domain to obj-object as type System.AppDomain
           catch
               display exception-object
           end-try

           set obj-object to obj-app-domain
      *> Casts can be used anywhere an expression of the target type can be used, e.g.
           display obj-object as type System.AppDomain::FriendlyName

      ********************************************************************************************************
      *
      *  U s i n g   t h e   N U L L   K e y w o r d
      *

      *> You can test whether an object reference is set to an instance of an object
      *> by comparing it with NULL. Object References are initialized to NULL by default.
           set obj-object to null

           if obj-object equals null
               display "It's definitely null"
           end-if

      ********************************************************************************************************
      *
      *  T h e   B o o l e a n   T y p e :   C O N D I T I O N - V A L U E
      *

      *> Inline declaration using 'declare' statement.
      *> An initial value can be specified using '=' .
           declare a-boolean as condition-value = false
      *> Condition-values can be set to true or false
           set a-boolean to true
           display a-boolean

      * They can be tested directly in conditional expressions
           if a-boolean
               display "OK"
           else
               display "wrong!"
           end-if

      ********************************************************************************************************
      *
      *  I n v o k i n g   m e t h o d s
      *

           set obj-string to "   Some text..."
           display obj-string

      *> You can invoke static or instance methods using an invocation expression.
      *> If the method has a returning item, then the expression can be used anywhere
      *> an expression of the return type is allowed, e.g. as the source of a set
      *> statement.
           set obj-string to obj-string::Replace("Some" "New")
           display obj-string

      *> You can also access properties using the same '::' syntax.
           display "The number of characters in the string is " obj-string::Length

      *> The methods being invoked may belong to classes in the same compilation unit.
      *> Use the 'invoke' verb if the method does not return anything...
           invoke type Class1::StaticMethod

           declare class1-obj as type Class1 = new Class1("Hello")

      *> The 'InstanceMethod' method is marked with the 'Obsolete' attribute
      *> and therefore the following statement produces a warning.
           invoke class1-obj::InstanceMethod


      ********************************************************************************************************
      *
      *  U s i n g   E n u m e r a t i o n s
      *
           display type System.DateTime::Today::DayOfWeek
           if type System.DateTime::Today::DayOfWeek equals type System.DayOfWeek::Friday
              display "Have a good weekend!"
           end-if

      *****************************************************************************************************
      *
      *  U s i n g   N e s t e d  c l a s s
      *
           declare encls-obj as type EnclosingClass = new EnclosingClass

           invoke encls-obj::CallNestedClsMethod()
           invoke encls-obj::EnclsInstanceMethod()

      ********************************************************************************************************
      *
      *  I m p l i c i t / E x p l i c i t  i m p l e m e n t a t i o n  o f  i n t e r f a c e
      *
           declare presentation = new Presentation
           declare printable as type IPrintable = presentation
           declare displayable as type IDisplayable = presentation

           invoke printable::WriteLine()
           invoke displayable::WriteLine()
           invoke presentation::WriteLine()

           declare presentationExplicit as type PresentationExplicit
                                                   = new PresentationExplicit
           set printable to presentationExplicit
           set displayable to presentationExplicit

           invoke printable::WriteLine()
           invoke displayable::WriteLine()
           invoke presentationExplicit::WriteLine()
       *> Inline casts to the interface type have the same effect:
           invoke presentationExplicit as type IPrintable::WriteLine()
           invoke presentationExplicit as type IDisplayable::WriteLine()
       

           display "Press the Enter key..."
           invoke type System.Console::ReadLine()

           goback.
       end program.


      ********************************************************************************************************
      *
      *  N o t e s   o n   C l a s s   D e f i n i t i o n s
      *

      *> Remember that the framework is case sensitive, so the compiler will treat the class name following
      *> class-id as case sensitive.  This class will be exposed as
      *> "NetCore.Class1" where the default namespace NetCore has been defined in the Project->Properties.
       class-id Class1
      *> If you wish to apply an attribute to a class definition use the 'attribute' keyword.
      *> In parentheses following the attribute name, you can specify any parameters
      *> to the attribute constructor (e.g. true in the first attribute), and then set attribute
      *> properties using the keyword 'property' (or the short form 'prop').
      *> When the name of the attribute ends with 'Attribute' then this portion of the name can be
      *> omitted, as in the use of WebServiceAttribute below.
      *> Note that the use of the CLSCompliant attribute produces a warning since it is only
      *> permitted when the containing assembly also has the CLSCompliant attribute.
           attribute System.CLSCompliantAttribute(true)
           attribute System.Web.Services.WebService(property Description = "A web service").

       working-storage section.

       *> instance and static data go here.
       01 str-obj string.
       01 static-string string static value "This is a static string".

      *>> <summary>
      *>> A simple definition of a static method. As for class-id, the name following method-id
      *>> is case sensitive.
      *>> </summary>
       method-id StaticMethod internal static.
      *> By default, methods in COBOL are public. Use the PRIVATE, PROTECTED, INTERNAL or
      *> PROTECTED INTERNAL modifiers to change this behaviour.  The INTERNAL modifier used above
      *> means that the method is only visible to classes in the same assembly (i.e. the same
      *> compilation unit).

          display static-string
       end method.

      *>> <summary>
      *>> Define a constructor for this class. Note that a default constructor will be generated
      *>> by the compiler if (and only if) no constructors are explicitly declared within your class.
      *>> Constructors are named "new", which can be specified in any combination of upper and lower case,
      *>> and should not have a returning item. Like other methods, they can be overloaded.
      *>> </summary>
       method-id New.
           set str-obj to "foo"
       end method.

       method-id New(lnk-str-obj as string).
      *> Note the specification of the method signature directly in the method header
      *> This removes the need for a PROCEDURE DIVISION header.
      *> Use the AS phrase to specify the parameter type.
           set str-obj to lnk-str-obj
       end method.

      *>> <summary>
      *>> Declare an instance method that is visible only to derived classes or classes within the
      *>> current assembly
      *>> </summary>
       method-id InstanceMethod protected internal
      *> Apply the obsolete attribute.  This means that attempts to reference this
      *> method will generate a warning.
      *> Specifying 'true' instead of 'false' as the second parameter to this attribute would
      *> cause references to this method to produce an error instead.
           attribute System.Obsolete("This function will be removed in a future version", false).

           display "In instance method"
           display "Instance data is " str-obj
           goback.
       end method.

       end class.
