      ********************************************************************************************************
      *
      * Copyright (C) Micro Focus 2010-2014.
      * All rights reserved.
      *
      *  This sample code is supplied for demonstration purposes only on an "as is" basis and "is for use at
      *  your own risk".
      *
      ********************************************************************************************************

       *>> <summary>
       *>> This class contains all the business logic and data for a book record:
       *>> It is used for reading, adding and deleting book data
       *>> </summary>
       class-id BookWrapper.Book.

       working-storage section.
       copy "book-rec-dotnet.cpy" replacing == (prefix) == by == book ==.

       01 bookFunction         pic x static.
           78 readRecord       value "1".
           78 addRecord        value "2".
           78 deleteRecord     value "3".
           78 nextRecord       value "4".
           78 allRecords       value "5".

       *>> <summary>
       *>> When the method is called, it reads book details
       *>> from a file by a given "stock number".
       *>> </summary>
       method-id Read static.
       local-storage section.
       01 file-status pic xx.
       procedure division using by value isbn-in as string
                          returning      myBook  as type BookWrapper.Book.

           set myBook to new BookWrapper.Book()
           set myBook::ISBN to isbn-in

           call "BookLegacy" using by value readRecord
                                   by value myBook::BookDetails
                                   by reference file-status

           invoke self::RaiseExceptionIfError(file-status)

       end method.

       *>> <summary>
       *>> This method records information about a book into the data file
       *>> </summary>
       *>> <param name="isbn-in">The stock number of the book</param>
       *>> <param name="title-in">The title of the book</param>
       *>> <param name="author-in">The author of the book</param>
       *>> <param name="type-in">The type of the book</param>
       *>> <param name="price-in">The price of the book</param>
       *>> <param name="onhand-in">The number of the books in stock</param>
       *>> <param name="sold-in">The number of books sold</param>
       *>> <returns></returns>
       method-id Add static.
       local-storage section.
       01 file-status pic xx.
       procedure division using by value  isbn-in as string
                                by value  stockno-in as string
                                by value  title-in   as string
                                by value  author-in  as string
                                by value  type-in    as string
                                by value  price-in   as decimal
                                by value  onhand-in  as binary-long
                                by value  sold-in    as binary-long
                                returning myBook     as type BookWrapper.Book.

           set myBook               to new BookWrapper.Book()
           set myBook::ISBN         to isbn-in
           set myBook::StockNumber  to stockno-in
           set myBook::Title        to title-in
           set myBook::Author       to author-in
           set myBook::Type         to type-in
           set myBook::RetailPrice  to price-in
           set myBook::NumberOnHand to onhand-in
           set myBook::NumberSold   to sold-in

           call "BookLegacy" using by value addRecord
                                   by value myBook::BookDetails
                                   by reference file-status

           invoke self::RaiseExceptionIfError(file-status)

       end method.

       *>> <summary>
       *>> Reads details about the next book following the given stock number
       *>> </summary>
       *>> <param name="isbn">The stock number of the last book</param>
       *>> <returns>Details about the next book</returns>
       method-id Next static.
       local-storage section.
       01 file-status pic xx.
       procedure division using by value isbn   as string
                          returning      myBook as type BookWrapper.Book.

           set myBook to new BookWrapper.Book()
           set myBook::ISBN to isbn

           call "BookLegacy" using by value nextRecord
                                   by value myBook::BookDetails
                                   by reference file-status

           invoke self::RaiseExceptionIfError(file-status)

       end method.

       *>> <summary>
       *>> Deletes a book from the data file given the book stock number
       *>> </summary>
       *>> <param name="stockno">The stock number of the book to delete</param>
       *>> <returns>The book that was deleted from the data file</returns>
       method-id Delete static.
       local-storage section.
       01 file-status pic xx.
       procedure division using by value isbn as string
                          returning      myBook  as type BookWrapper.Book.

           set myBook to new BookWrapper.Book()
           set myBook::StockNumber to isbn

           call "BookLegacy" using by value deleteRecord
                                   by value myBook::BookDetails
                                   by reference file-status

           invoke self::RaiseExceptionIfError(file-status)

       end method.

       *>> <summary>
       *>> Raises a specific exception given a file status
       *>> </summary>
       *>> <param name="file-status">The file status that determines the exception to raise</param>
       method-id RaiseExceptionIfError static.
       linkage section.
       01 file-status pic xx.
       procedure division using file-status.
           if file-status <> "00" and file-status <> "02"
               evaluate file-status
                   when "35"
                       raise new BookWrapper.BookException(type BookWrapper.ExceptionCodes::DataFileNotFound)
                   when "23"
                       raise new BookWrapper.BookException(type BookWrapper.ExceptionCodes::StockItemNotFound)
                   when "46"
                       raise new BookWrapper.BookException(type BookWrapper.ExceptionCodes::NoMoreItemsLeft)
                   when "99"
                       raise new BookWrapper.BookException(type BookWrapper.ExceptionCodes::ItemAlreadyExists)
                   when other
                       raise new BookWrapper.BookException(type BookWrapper.ExceptionCodes::FileError)
               end-evaluate
           end-if
       end method.


       *>> <summary>
       *>> Returns a pointer to the book details
       *>> </summary>
       *>> <returns>Pointer to the book details</returns>
       property-id BookDetails pointer.
       getter.
           set property-value to address of book-details
       end property.

       *>> <summary>
       *>> Returns the computed stock value of the book.
       *>> </summary>
       *>> <returns>The computed stock value of the book</returns>
       property-id StockValue float-short.
       getter.
           compute property-value = book-retail * book-onhand
       end property.
       end class.
