package com.ibm.ie.reeng.rt.common.comms;

import com.ibm.ie.reeng.rt.common.Comms;
import com.ibm.ie.reeng.rt.common.Marshall;
import com.ibm.ie.reeng.rt.common.Util;
import com.ibm.ie.reeng.rt.common.Log;

import com.ibm.ctg.client.JavaGateway;
import com.ibm.ctg.client.ECIRequest;

import java.io.IOException;
import java.io.UnsupportedEncodingException;

import java.net.UnknownHostException;



public class CtgComms implements Comms
{
    private static String jgateURL;       
    private static String serverName;
    private static String routerName;
    private static String serverCodepage;
    private static int commAreaSize;
    private static boolean debug;
    
    public static boolean initialised = false;

    private String userName;
    private String password;
    private String sessionId;

    private byte[] emptyCommarea;
    private JavaGateway jgaConnection;
    private byte[] commArea;

    private final int COMMAREA_MAX_LENGTH = 32045;


    protected void initCtgConfigData(String jgateURL,
                                     String serverName,
                                     String router,
                                     int commareaSize,
                                     String serverCodepage,
                                     boolean debug)
    {    
        initialised = true;

        this.jgateURL = jgateURL;
        this.serverName = serverName;
        this.routerName = router;
        this.commAreaSize = commareaSize;
        
        if(this.commAreaSize <= 0)
            this.commAreaSize = COMMAREA_MAX_LENGTH;
        
        this.serverCodepage = serverCodepage;
        this.debug = debug;


        // check for required properties.
        if (jgateURL == null)
            fatalError("property jgate not set");
        if (serverName == null)
            fatalError("property cics server not set");
        if (router == null)
            fatalError("property cics router not set");
        if (serverCodepage == null)
            fatalError("property server codepage not set");
    }


    protected void initCtgConnection(String sessionId,
                                     String userName,
                                     String password)
    {

        this.userName = userName;
        this.password = password;
        this.sessionId = sessionId;

        try
        {
            jgaConnection = new JavaGateway();
            jgaConnection.setURL(jgateURL);
            jgaConnection.open();
        }
        catch (IOException e) { fatalException(e); }

        sessionId = "+"
            + Util.padLeft(String.valueOf(sessionId), 10, Util.ZERO);

        commArea = new byte[commAreaSize];

        emptyCommarea = createSpaceFilledCommarea();

        if (debug)
        {
            information("initialization complete:");
            information("  jgateURL:       " + jgateURL);
            information("  userName:       " + userName);
            information("  password:       " + password);
            information("  serverName:     " + serverName);
            information("  routerName:         " + routerName);
            information("  commAreaSize:   " + commAreaSize);
            information("  serverCodepage: " + serverCodepage);
        }    
    }


    /**
     * Invokes a server rule using the CICS transaction gateway.
     * The format of the commarea is as follows:
     * <UL>
     * <LI>0...1 - character version identifier.
     * <LI>2...12 - numeric session identifier.
     * <LI>13...18 - character rule name.
     * <LI>19...28 - numeric COMMAREA size. (Why?)
     * <LI>29...38 - numeric body size. (Why?)
     * <LI>39...42 - character padding &quot;001N&quot;. (Why?)
     * <LI>43... - body (marshalled input or output)
     * </UL>
     * I assume that this simply reflects the router code because some
     * of it seems redundant.
     */
    public void send(String ruleName, Marshall marshall)
    {
        try { trySend(ruleName, marshall); }
        catch (Exception e) { fatalException(e); }
    }

    /**
     * Invokes a server rule using the CICS transaction gateway.
     * The format of the commarea is as follows:
     * <UL>
     * <LI>0...1 - character version identifier.
     * <LI>2...12 - numeric session identifier.
     * <LI>13...18 - character rule name.
     * <LI>19...28 - numeric COMMAREA size. (Why?)
     * <LI>29...38 - numeric body size. (Why?)
     * <LI>39...42 - character padding &quot;001N&quot;. (Why?)
     * <LI>43... - body (marshalled input or output)
     * </UL>
     * I assume that this simply reflects the router code because some
     * of it seems redundant.
     */
    private void trySend(String ruleName, Marshall marshall)
    throws UnsupportedEncodingException, IOException
    {
        StringBuffer buf = new StringBuffer();

        int lengthToSend = 10000; // Default size which works in most cases
    
    	//GQ - One of the rules in broker/branch sends 26k of data and the original hard coded value of 10000 was causing the
		//Cics GWay to crash. Changed this to body.length() which also caused the CIC's to abend. 
        //Can't explain why body.length was causing CICs to abend. 
        if (buf.length() > 10000)
          lengthToSend = commAreaSize;
        
        // add the header:
        buf.append("0I");
        buf.append(sessionId);
        buf.append(Util.padRight(ruleName, 8, Util.SPACE));
        buf.append(padInt(commAreaSize, 10));
//      buf.append(padInt(body.length(), 10));
        buf.append(padInt(lengthToSend, 10));
        buf.append(padInt(1, 3) + "N");
        int headerLength = buf.length();

        // add the body.
        String body = marshall.getWriteBuffer();
        int bodyLength = body.length();
        buf.append(body);

        if (debug)
        {
            information("sending CTG ECI request:");
            information("serverName: " + serverName);
            information("userName: " + userName);
            information("password: " + password);
            information("routerName: " + routerName);
            logCommarea(buf.toString());
        }
       
        // fill the comm area.
        System.arraycopy(emptyCommarea, 0, commArea, 0, commAreaSize);
        byte[] hostCommArea = buf.toString().getBytes(serverCodepage);
        if (hostCommArea.length > commAreaSize)
            fatalError("message size too big for COMMAREA");
        System.arraycopy(hostCommArea, 0, commArea, 0, hostCommArea.length);

        if (debug)
            logHostCommarea(commArea);

        ECIRequest eciRequest
            = new ECIRequest(serverName,       // CICS Server
                            userName,          // UserId
                            password,          // Password
                            routerName,        // Program name
                            commArea,          // Commarea
                            ECIRequest.ECI_NO_EXTEND,
                            ECIRequest.ECI_LUW_NEW);

        jgaConnection.flow(eciRequest);

        if (eciRequest.getRc() != 0) requestError(eciRequest, "flow failed");

        String returnCommarea = new String(commArea, serverCodepage);

        if (debug)
        {
            information("received CTG ECI response:");
            information("return code:" + eciRequest.getRc());
            information("commarea result code field =  "
                + returnCommarea.substring(1, 4));
            logCommarea(returnCommarea);
            logHostCommarea(commArea);
        }

        if (!returnCommarea.substring(1, 4).equals("000"))
        {
            requestError(eciRequest, "Message Header is: " 
                   + returnCommarea.substring(0, headerLength));
        }

        if (debug) information("CTG comms completed successfully");

        marshall.setReadBuffer(returnCommarea.substring(headerLength));
    }

    private byte[] createSpaceFilledCommarea()
    {
        byte[] bytes = new byte[commAreaSize];
        byte[] someSpaces = null;
        try
        {
            someSpaces = ("                                  "
                + "                                                   "
                + "                                                   "
                + "                                                   "
                + "                                                   "
                + "                                                   "
                + "                                                   "
                + "                                                   "
                + "                                                   "
                + "                                                   "
                ).getBytes(serverCodepage);
        }
        catch (Exception e)
        {
            fatalException(e);
        }

        int spacesLength = someSpaces.length;
        int last = commAreaSize - spacesLength;

        int i = 0;
        for (; i <= last ; i += spacesLength)
        {
            System.arraycopy(someSpaces, 0, bytes, i, spacesLength);
        }

        System.arraycopy(someSpaces, 0, bytes, i, commAreaSize - i);
        return bytes;
    }

    private static String padInt(int n, int width)
    {
        return Util.padLeft(String.valueOf(n), width, Util.ZERO);
    }


    private void logCommarea(String buf)
    {
        if (buf.length() > 160)
            buf = buf.substring(0, 160) + "...";
        information("CLIENT COMMAREA: " + buf);
    }

    private void logHostCommarea(byte[] bytes)
    {
        information("HOST COMMAREA (first 256 bytes): ");
        int last = Math.min(256, (bytes.length / 32) * 32);
        for (int i = 0; i < last; i += 32)
        {
            StringBuffer buf = new StringBuffer();
            buf.append(Util.padLeft(String.valueOf(i), 6, ' '));
            buf.append(": ");
            int b;
            for (int j = i; j < i + 32; j++)
            {
                b = bytes[j] >= 0 ? bytes[j] : (int)256 + bytes[j];
                buf.append(hexDigits[b / 16]);
                buf.append(hexDigits[b % 16]);
            }
            information(buf.toString());
        }
    }
   
    private static String[] hexDigits = new String[] { "0", "1", "2", 
        "3", "4", "5", "6", "7", "8", "9", "A", "B", "C", "D", "E", "F" };
    
    public void close()
    {
        try
        {
            jgaConnection.close();
        }
        catch (Exception e) { e.printStackTrace(); }
        finally
        {
            jgaConnection = null;
            commArea = null;
        }
    }

    protected void finalize() throws Throwable { close(); }

    protected void information(String s){}

    protected void requestError(ECIRequest req, String extra){}

    protected void fatalError(String s){}

    protected void fatalException(Exception e){}


}