package com.ibm.ie.reeng.rt.common;

import java.lang.reflect.Field;

import java.text.DecimalFormat;
import java.text.NumberFormat;

import java.util.HashSet;
import java.util.HashMap;
import java.util.Locale;

import javax.swing.Action;
import javax.swing.UIManager;
import javax.swing.text.JTextComponent;
import javax.swing.JTextField;
import javax.swing.KeyStroke;

import java.rmi.server.UID;
import java.net.UnknownHostException;


/**
 * This class provides sundry utility functions for the runtime and the
 * generated code.
 */
public class Util
{
    // Note: MIN_INTEGER is not -0x80000000 as might be
    // expected - this value appears to be reserved.
    public final static int MAX_INTEGER =  0x7FFFFFFF;
    public final static int MIN_INTEGER = -0x7FFFFFFF;

    public final static int MAX_SMALLINT =  0x7FFF;
    public final static int MIN_SMALLINT = -0x8000;

    public final static int MAX_TIME = 86399999; // Milliseconds in day.

    public final static char SPACE = ' ';
    public final static char ZERO = '0';


    /** Speed takes precedence over simple code for padding. */
    public static String padLeft(String string, int len, char pad)
    {
        int actual = string.length();

        if (actual == len) return string;
        if (actual > len) return string.substring((actual - len), actual);
        else
        {
            char[] destination = new char[len];
            char[] source = string.toCharArray();
            int i;

            len -= actual;

            for (i = 0; i < len; i++) destination[i] = pad;

            int j = 0;

            while (j < actual) destination[i++] = source[j++];

            return new String(destination);
        }
    }


    /** Speed takes precedence over simple code for padding. */
    public static String padRight(String string, int len, char pad)
    {
        int actual = string.length();

        if (actual == len) return string;
        if (actual > len) return string.substring(0, len);
        else
        {
            char[] destination = new char[len];
            char[] source = string.toCharArray();
            int i;

            len -= actual;

            for (i = 0; i < actual; i++) destination[i] = source[i];

            int j = 0;

            while (j++ < len) destination[i++] = pad;

            return new String(destination);
        }
    }


    public static String unpadRight(String string, char pad)
    {
        int i = string.length() - 1;

        while (i >= 0 && string.charAt(i) == pad) i--;

        return string.substring(0, (i + 1));
    }

    public static final void unpadRight(StringBuffer buf, char pad)
    {
        int i = buf.length() - 1;
        while (i >= 0 && buf.charAt(i) == pad) i--;
        buf.setLength(i);
    }

    public static String unpadLeft(String string, char pad)
    {
        int i = 0;

        while (i < string.length() && string.charAt(i) == pad) i++;

        return string.substring(i, string.length());
    }


    // ------------------------------------------------------------------------


    private static DecimalFormat decimal;


    static
    { 
        NumberFormat formatter = NumberFormat.getNumberInstance(Locale.UK);

        if (!(formatter instanceof DecimalFormat))
            Log.fatalError("Util", "NumberFormat of unexpected type " +
                formatter.getClass().getName());

        decimal = (DecimalFormat)formatter;
        decimal.applyPattern("#0.0");
        decimal.setMaximumFractionDigits(Integer.MAX_VALUE);
    }


    /**
     * Double.toString() uses computerized scientific notation for very
     * large and small numbers. This routine does not use an exponential
     * format no matter how large or small the number given is.
     */
    public static String toDoubleString(double d)
    {
        return decimal.format(d);
    }


    // ------------------------------------------------------------------------


    public static Object getClearValue(Object object)
    {
        if (object instanceof Integer) return new Integer(0);
        else if (object instanceof Double) return new Double(0);
        else if (object instanceof String) return "";
        else if (object instanceof java.sql.Date)
            return TemporalFunctions.ZERO_DATE;
        else if (object instanceof java.sql.Time)
            return TemporalFunctions.ZERO_TIME;
        else if (object instanceof java.sql.Timestamp)
            return TemporalFunctions.ZERO_TIMESTAMP;
        else
        {
            Log.fatalError("Util",
                "can not clear " + object.getClass().getName());

            return null;
        }
    }


    // ------------------------------------------------------------------------


    private static HashMap nameToAction;


    /**
     * Use this function to get the copy and paste actions.
     * <p>
     * Note: copy in JDK 1.2 can only copy the entire contents of a text
     * field.
     */
    public static Action getAction(String name)
    {
        if (nameToAction == null)
        {
            nameToAction = new HashMap();

            JTextField field = new JTextField();
            Action[] actions = field.getActions();

            for (int i = 0; i < actions.length; i++)
                nameToAction.put(actions[i].getValue(Action.NAME), actions[i]);
        }

        return (Action)(nameToAction.get(name));
    }


    // ------------------------------------------------------------------------


    private static HashMap actionToKeyStrokes;


    /**
     * JDK BUG WORK AROUND.
     * <p>
     * This function provides the same functionality as
     * JTextComponent.getKeyStrokesForAction() should - because as
     * of JDK 1.2 this function is still unimplemented.
     */
    public static KeyStroke[] getKeyStrokesForAction(Action action)
    {
        if (actionToKeyStrokes == null)
        {
            actionToKeyStrokes = new HashMap();

            Object o = UIManager.get("TextField.keyBindings");

            if (!(o instanceof JTextComponent.KeyBinding[])) return null;

            JTextComponent.KeyBinding[] bindings =
                (JTextComponent.KeyBinding[])o;

            for (int i = 0; i < bindings.length; i++)
            {
                KeyStroke keyStroke = bindings[i].key;
                Action keyStrokeAction = getAction(bindings[i].actionName);

                KeyStroke[] list =
                    (KeyStroke[])actionToKeyStrokes.get(keyStrokeAction);

                if (list == null) list = new KeyStroke[1];
                else
                {
                    KeyStroke[] newList = new KeyStroke[list.length + 1];

                    System.arraycopy(list, 0, newList, 0, list.length);
                    list = newList;
                }

                list[list.length - 1] = keyStroke;

                actionToKeyStrokes.put(keyStrokeAction, list);
            }
        }

        return (KeyStroke[])actionToKeyStrokes.get(action);
    }

    /**
     * This is influenced by the style of
     * javax.swing.JComponent.toString().  It is used in the
     * implementation of toString() for views and rules.
     */
    public static String toString(Object o)
    {
        return toString(o, 1);
    }

    /**
     * The recursive depth controls the level to which members of the
     * Object are also converted into a readable form.
     * @param   o   the Object to converted into a readable form.
     * @param   depth   the recursive depth.
     */
    public static String toString(Object o, int depth)
    {
        StringBuffer buf = new StringBuffer();
        writeTo(buf, o, depth - 1);
        return buf.toString();
    }

    // ----------- some utility methods for reflection.

    public static boolean isPrimitiveWrapper(Class c)
    {
        return wrapperClasses.contains(c);
    }

    public static boolean considerAtomic(Class c)
    {
        return primitiveTypes.contains(c)
            || wrapperClasses.contains(c)
            || simpleClasses.contains(c);
    }

    /**
     * This holds the java.lang.Class instances of types which are
     * considered to be simple, atomic types.
     */
    private static HashSet primitiveTypes = new HashSet();
    private static HashSet wrapperClasses = new HashSet();
    private static HashSet simpleClasses = new HashSet();
    static
    {
        primitiveTypes.add(Boolean.TYPE);
        primitiveTypes.add(Byte.TYPE);
        primitiveTypes.add(Character.TYPE);
        primitiveTypes.add(Double.TYPE);
        primitiveTypes.add(Float.TYPE);
        primitiveTypes.add(Integer.TYPE);
        primitiveTypes.add(Long.TYPE);
        primitiveTypes.add(Short.TYPE);
        primitiveTypes.add(Void.TYPE);

        wrapperClasses.add(Boolean.class);
        wrapperClasses.add(Byte.class);
        wrapperClasses.add(Character.class);
        wrapperClasses.add(Double.class);
        wrapperClasses.add(Float.class);
        wrapperClasses.add(Integer.class);
        wrapperClasses.add(Long.class);
        wrapperClasses.add(Short.class);
        wrapperClasses.add(Void.class);

        simpleClasses.add(String.class);
        simpleClasses.add(java.sql.Date.class);
        simpleClasses.add(java.sql.Time.class);
        simpleClasses.add(java.sql.Timestamp.class);
    }

    private static void writeTo(StringBuffer buf, Object o, int depth)
    {
        try
        {
            Class c = o.getClass();

            buf.append(c.getName());
            buf.append(" [");

            Field[] fields = c.getFields();

            boolean atStart = true;
            for (int i = 0; i < fields.length; i++)
            {
                if (!atStart) buf.append(", ");

                Class type = fields[i].getType();

                buf.append(fields[i].getName());
                buf.append(" = ");
                Object fieldVal = fields[i].get(o);
                if (type == String.class)
                {
                    buf.append("\"");
                    buf.append(fieldVal.toString());
                    buf.append("\"");
                }
                else if (considerAtomic(type))
                {
                    buf.append(fieldVal.toString());
                }
                else
                {
                    if (fieldVal == null) buf.append("null");
                    else if (type.isArray())
                    {
                        buf.append(type.getComponentType().getName());
                        buf.append("[");
                        buf.append(Integer.toString(((Object[])fieldVal).length));
                        buf.append("]");
                    }
                    else if (depth <= 0)
                    {
                        buf.append("<");
                        buf.append(type.getName());
                        buf.append(" instance");
                        buf.append(">");
                    }
                    else // depth > 0
                    {
                        writeTo(buf, fieldVal, depth - 1);
                    }
                }
                atStart = false;
            }
            buf.append("]");
        }
        catch (Exception e)
        {
            Log.fatalException(e);
        }
    }

    // ----------------------------------------------------------------

    /**
     * Tricky to implement. This is not fully correct but there isn't
     * any need to be overly paranoid.
     */
    public static long generateGUID()
    {
        // unique with respect to the host:
        int uid = new UID().toString().hashCode();

        int hostID = 0;
        try
        {
            // unique per host:
            hostID = java.net.InetAddress.getLocalHost().hashCode();
        }
        catch (UnknownHostException e)
        {
            // I guess we're not connected to a network so it shouldn't
            // matter.
        }

        return ((long)hostID + Integer.MIN_VALUE) << 32
            | ((long)uid + Integer.MIN_VALUE);
    }

    /**
     * This system component is not fully implemented. It is not
     * possible to build an exhaustive list of Java platforms. Nor are
     * all known Hps values for execmode known. Currently (without
     * confirmation), we think the list includes:
     * <UL>
     * <LI> CICS
     * <LI> IMS
     * <LI> BATCH
     * <LI> UNKNOW
     * <LI> PC DOS
     * <LI> PC OS2
     * <LI> WIN 95
     * <LI> WIN NT
     * <LI> W4
     * <LI> AIX
     * <LI> OS2
     * <LI> WINNT
     * <LI> HP-UX
     * <LI> NCR
     * <LI> OS400
     * <LI> SUNOS
     * <LI> CICS
     * </UL>
     * However these values do not seem to be mutually exclusive. The
     * mapping will have to be tuned/adjusted with experience.
     * <p>
     * A list of values for the Java &quot;os.name&quot; system property
     * was found <A HREF="http://www.tolstoy.com/samizdat/sysprops.html">here</A>.
     * Obviously this list is also prone to revision and expansion.
     */
    public static String getExecutionEnvironment()
    {
        if (executionEnvironment == null)
        {
            String key = System.getProperty("os.name").toLowerCase();

            if (key.equals("windowsnt")) executionEnvironment = "WIN NT";
            else if (key.equals("windows nt")) executionEnvironment = "WIN NT";
            else if (key.equals("windows 95")) executionEnvironment = "WIN 95";
            else if (key.equals("windows 2000")) executionEnvironment = "WIN NT";
            else if (key.equals("windows2000")) executionEnvironment = "WIN NT";
            else if (key.equals("os/2")) executionEnvironment = "PS OS2";
            else if (key.equals("hp-ux")) executionEnvironment = "HP-UX";
            else if (key.equals("aix")) executionEnvironment = "AIX";
            else if (key.equals("solaris")) executionEnvironment = "SUNOS";
            else
            {
                Log.error("Util", "unknown platform: " + key);
                executionEnvironment = "UNKNOW";
            }
        }

        return executionEnvironment;
    }

    private static String executionEnvironment;
}
