package com.ibm.ie.reeng.rt.common;


import java.text.SimpleDateFormat;




/**
 * This differs from DefaultMarshall by marshalling Dates, Times and
 * Timestamps using the ISO character based format.
 */
public class RabobankMarshall extends DefaultMarshall
{
    private final static int SMALLINT_DIGITS = 5;
    private final static int INTEGER_DIGITS = 10;
    
    
    // these are all initialized below.
    private static String DATE_FORMAT;
    private static int DATE_FORMAT_LENGTH;
    private static SimpleDateFormat DATE;

    //Format for MM/DD/YYYY
    private static String DATE_SLASH_FORMAT;
    private static int DATE_SLASH_FORMAT_LENGTH;
    private static SimpleDateFormat DATE_SLASH;

    private static String TIME_FORMAT;
    private static int TIME_FORMAT_LENGTH;
    private static SimpleDateFormat TIME;

    // Note that the java.text.SimpleDateFormat class cannot format sub
    // millisecond precision. Therefore we logically split timestamps
    // into a Datetime portion and a fraction portion.
    private static String DATETIME_FORMAT;
    private static int DATETIME_FORMAT_LENGTH;
    private static SimpleDateFormat DATETIME;
    private static int FRACTION_DIGITS;
    private static int NANOS_PER_FRACTION_UNIT;
    private static int TIMESTAMP_FORMAT_LENGTH;

    /**
     * It's slightly easier to read by putting all the initialization
     * here. It also reduces the dependency on initialization ordering.
     */
    static
    {
        DATE_FORMAT = "yyyy-MM-dd";
        DATE_FORMAT_LENGTH = DATE_FORMAT.length();
        DATE = new SimpleDateFormat(DATE_FORMAT);

        DATE_SLASH_FORMAT = "MM/dd/yyyy";
        DATE_SLASH_FORMAT_LENGTH = DATE_SLASH_FORMAT.length();
        DATE_SLASH = new SimpleDateFormat(DATE_SLASH_FORMAT);

        TIME_FORMAT = "HH:mm:ss";
        TIME_FORMAT_LENGTH = TIME_FORMAT.length();
        TIME = new SimpleDateFormat(TIME_FORMAT);

        DATETIME_FORMAT = "yyyy-MM-dd-HH.mm.ss";
        DATETIME_FORMAT_LENGTH = DATETIME_FORMAT.length();
        DATETIME = new SimpleDateFormat(DATETIME_FORMAT);

        FRACTION_DIGITS = 6;       // ISO compliant
        NANOS_PER_FRACTION_UNIT = 1000;     // i.e. 10^(9 - FRACTION_DIGITS)
        TIMESTAMP_FORMAT_LENGTH = DATETIME_FORMAT_LENGTH + 1 + FRACTION_DIGITS;
    }

    public Marshall writeDate(java.sql.Date field)
    {
        put(DATE.format(field));
        return this;
    }

    public Marshall writeTime(java.sql.Time field)
    {
        put(TIME.format(field));
        return this;
    }

    public Marshall writeTimestamp(java.sql.Timestamp field)
    {
        put(DATETIME.format(field));
        put(".");
        String fract
            = Integer.toString(field.getNanos() / NANOS_PER_FRACTION_UNIT);
        put(Util.padLeft(fract, FRACTION_DIGITS, Util.ZERO));

        return this;
    }

    public java.sql.Date readDate()
    {
        try
        {
            String string = get(DATE_FORMAT_LENGTH);

                if (string.trim().equals(""))
                return TemporalFunctions.ZERO_DATE;
                if (string.charAt(2) == '/') 
                    return new java.sql.Date(DATE_SLASH.parse(string).getTime());
              
            return new java.sql.Date(DATE.parse(string).getTime());
        }
        catch (Exception e)
        {
            Log.fatalException(e);
            return null;    // keep the compiler happy
        }
    }

    public java.sql.Time readTime()
    {
        try
        {
            String string = get(TIME_FORMAT_LENGTH);
                if (string.trim().equals(""))
                return TemporalFunctions.ZERO_TIME; 
            return new java.sql.Time(TIME.parse(string).getTime());
        }
        catch (Exception e)
        {
            Log.fatalException(e);
            return null;    // keep the compiler happy
        }
    }

    public java.sql.Timestamp readTimestamp()
    {
        String string = get(DATETIME_FORMAT_LENGTH);
        if (string.trim().equals(""))
            return TemporalFunctions.ZERO_TIMESTAMP;
        try
        {
            java.util.Date date = DATETIME.parse(string);

            java.sql.Timestamp ts = new java.sql.Timestamp(date.getTime());

            get(1); // discard the period.
            string = get(FRACTION_DIGITS);
            int nanos = NANOS_PER_FRACTION_UNIT * Integer.parseInt(string);

            ts.setNanos(nanos);

            return ts;
        }
        catch (Exception e)
        {
            Log.fatalException(e);
            return null;    // keep the compiler happy
        }
    }
    
    
    public int readInteger()
    {
        return getInteger(INTEGER_DIGITS);
    }

    public int readSmallint()
    {
        return getInteger(SMALLINT_DIGITS);
    }
    
    public int getInteger(int digits)
    {
        String string = get(digits + 1).trim();

        if (string.equals("")) return 0;
        
        boolean negative = (string.charAt(0) == '-');

        string = string.substring(1, string.length());
        string = Util.unpadLeft(string, Util.ZERO);

        if (string.length() == 0) return 0;

        int field = Integer.parseInt(string);

        if (negative) field *= -1;

        return field;
    }
    
    public double readDec(int length, int fraction)
    {
        // Note: the field length of DEC data items does not include the sign.

        String string = get(length + 1);
        boolean minus = (string.charAt(0) == '-');

        string = string.substring(1, string.length()); // Remove sign byte.

        if (string.trim().equals(""))
            return 0;

        if (fraction > 0)
        {
            if (fraction == length) string = "0." + string;
            else
            {
                String integer = string.substring(0, (length - fraction));

                String decimal = string.substring((length - fraction),
                    string.length());

                string = integer + "." + decimal;
            }
        }

        double field = parseDouble(string);

        if (minus) field *= -1;

        return field;
    }
    
    private double parseDouble(String string)
    {
        string = Util.unpadLeft(string, Util.ZERO);

        if (string.length() == 0) return 0;

        return Double.valueOf(string).doubleValue();
    }
}
