package com.ibm.ie.reeng.rt.common;

/**
 * This is the base class for all generated "proxy rules". As the name
 * would suggest, subclasses act as proxies for remote rules.
 * <p>
 * This class depends on a number of runtime properties:
 * <UL>
 * <LI>"server.marshall" - fully qualified name of the class used to 
 *      marshall/unmarshall messages to the server. This class must 
 *      extend {@link Marshall}.
 * <LI>"server.comms" - fully qualified name of the class used to 
 *      send a request to the server. This class must implement {@link Comms}.
 * <LI>"trace.requests" - if true log all requests sent to the server.
 * <LI>"trace.responses" - if true log all responses received from the server.
 * <LI>"trace.stack" - write a stack trace to the log every time this
 *      rule is used.
 * </UL>
 *
 */
public abstract class ProxyRule
{
    private static boolean dumpRequest;
    private static boolean dumpResponse;
    private static boolean dumpStack;
    private static Class marshallClass;
    private static Comms comms;

    /**
     * Initialize all the statics here.
     */
    static
    {
        // get the debug settings.
        dumpRequest = Config.instance().getBoolean("trace.requests");
        dumpResponse = Config.instance().getBoolean("trace.responses");
        dumpStack = Config.instance().getBoolean("trace.stack");

        // Get the Marshall class.
        String marshallClassName
            = Config.instance().getString("server.marshall");
        
        if (marshallClassName == null)
        {
            die("\"server.marshall\" system property is not set.");
        }

        try { marshallClass = Class.forName(marshallClassName); }
        catch (ClassNotFoundException e)
        {
            die("cannot load Marshall class " + marshallClassName);
        }
        catch (Exception e)
        {
            die("error creating Marshall class " + marshallClassName
                + ": " + e);
        }

        // Get a comms instance.
        String name = Config.instance().getString("server.comms");

        if (name == null)
        {
            die("\"server.comms\" system property is not set. ");
        }

        try
        {
            comms = (Comms)Class.forName(name).newInstance();
        }
        catch (ClassNotFoundException e)
        {
            die("cannot load Comms class " + name);
        }
        catch (Exception e)
        {
            die("error creating instance of Comms class " + name);
        }
    }

    private static void die(String message)
    {
        Log.fatalError("ProxyRule", message);
    }

    protected ProxyRule()
    {
    }

    /** Subclasses override this to return the input view. */
    protected View getInputView() { return null; }

    public void clearInputView() { }

    /** 
     * Subclasses override this to return the server response to
     * the output view.
     */
    protected View getOutputView() { return null; }


    /** 
     * Subclasses implement this to specify the name of the remote rule
     * to invoke. Normally this will return the Hps shortname of the
     * rule.
     */
    protected abstract String getRemoteName();

    private Marshall createMarshall()
    {
        try
        {
            return (Marshall)marshallClass.newInstance();
        }
        catch (Exception e)
        {
            Log.fatalError(this,
                "error creating instance of Marshall class " +
                marshallClass);
            return null; // keep the compiler happy.
        }
    }

    public void invokeRemote()
    {
        Log.information(this, "entering proxy rule " + getRemoteName());
        
        if (dumpStack) Location.logStack();

        Marshall marshall = createMarshall();

        if (getInputView() != null) getInputView().writeTo(marshall);

        if (dumpRequest) marshall.dumpBuffer();

        // int len = getOutputView().writeTo(new SizeofMarshall()).getLength();

        comms.send(getRemoteName(), marshall);

        if (dumpResponse) marshall.dumpBuffer();

        if (getOutputView() != null) getOutputView().readFrom(marshall);

        Log.information(this, "exiting proxy rule " + getRemoteName());
    }

    public String toString()
    {
        return Util.toString(this, 2);
    }
}
