package com.ibm.ie.reeng.rt.common;

import java.util.StringTokenizer;
import java.io.InputStream;


/**
 * This class is responsible for loading the runtime properties for a
 * rebased client. It reads them from the "runtime.properties" resource.
 * After reading this resource, it checks for the existance of the
 * "properties.files" property. This is treated as a comma separated
 * list of property resource names sans the .properties suffix.
 * <p>
 * Note that this class stores the loaded properties in the system
 * properties; this is required to allow the user to set properties
 * which are used by the standard library like "user.timezone", etc. in
 * their runtime.properties file. 
 * <P>
 * Note this class does not use com.ibm.ie.reeng.rt.common.Log as doing
 * so introduces a circular dependency.
 */
public class PropertyConfigSource implements Config.Source
{
    private static String PROPERTIES_RESOURCE_NAME = "runtime";

    public PropertyConfigSource()
    {
        loadProperties(PROPERTIES_RESOURCE_NAME);
        
        String otherFiles = getString("properties.files");

        if (otherFiles != null)
        {
            StringTokenizer toks = new StringTokenizer(otherFiles, ",");
            while (toks.hasMoreTokens())
            {
                loadProperties(toks.nextToken().trim());
            }
        }
    }

    private static void loadProperties(String resourceName)
    {
        String fileName = resourceName + ".properties";
        // open the property file resource
        InputStream in = ClassLoader.getSystemResourceAsStream(fileName);

        try
        {
            System.getProperties().load(in);
            System.out.println(fileName + " loaded");
        }
        catch (Exception e) { e.printStackTrace(); }
    }

    public String getString(String key)
    {
        return System.getProperty(key);
    }

    public boolean getBoolean(String key)
    {
        return Boolean.getBoolean(key);
    }

    public Integer getInteger(String key)
    {
        return Integer.getInteger(key);
    }

    public int getInt(String key)
    {
        Integer i = getInteger(key);

        if (i != null) return i.intValue();

        System.out.println("Config Error: " + key
            + " is not set to a valid integer.");
        return -1;
    }
}
