package com.ibm.ie.reeng.rt.common;

/**
 * This class acts as an abstract base class for runtime classes which
 * require knowledge of the original view structure of the Hps
 * application from which the rebased application was generated.
 * The generated classes corresponding to Hps views interact with
 * instances of this type through <CODE>readFrom()</CODE> and
 * <CODE>writeTo()</CODE> methods. These methods directly encode the
 * structure of the original views and in particular the Hps types of
 * contained fields.
 * <P>
 * A design goal of the rebase tool was to use standard Java types to
 * represent data instead of designing custom classes as analogues of
 * the Hps types and as much as possible to use standard Java functions
 * and operators. The type mapping is as follows:
 * <TABLE>
 * <TR><TD><B>Original Hps Type</B><TD><B>Rebased Java Type</B>
 * <TR><TD>CHAR(n)                 <TD>java.lang.String
 * <TR><TD>VARCHAR(n)              <TD>java.lang.String
 * <TR><TD>SMALLINT                <TD>int
 * <TR><TD>INTEGER                 <TD>int
 * <TR><TD>DEC(n,m)                <TD>double
 * <TR><TD>PIC s                   <TD>double
 * <TR><TD>DATE                    <TD>java.sql.Date
 * <TR><TD>TIME                    <TD>java.sql.Time
 * <TR><TD>TIMESTAMP               <TD>java.sql.Timestamp
 * </TABLE>
 * <P>
 * From the table, it is obvious that the mapping from Hps types to Java
 * types is not bijective; i.e. given the Java type of a field we cannot
 * infer the original Hps type. It would be nice to be able to
 * completely forget about the original types in the rebased code but
 * unfortunately there are a number of areas of functionality which
 * require knowlege of the of the original types:
 * <UL>
 * <LI> Communications with the server. The server expects views to be
 *      packaged to reflect the size and type of the original Hps.
 * <LI> Implementation of <CODE>overlay</CODE>.
 *      This has to reflect the Hps semantics which is completely
 *      dependant on the memory layout of the original Hps based data.
 * <LI> Implementation of <CODE>sizeof</CODE>. For the same reason.
 * </UL>
 * All of this functionality is implemented by passing an appropriate
 * instance of <CODE>Marshall</CODE> to the <CODE>readFrom()</CODE> and
 * <CODE>writeTo()</CODE> methods of the generated views.
 */
public abstract class Marshall
{
    /** Represents the writing state. */
    protected final static int WRITE = 0;

    /** Represents the reading state. */
    protected final static int READ = 1;

    /**
     * Indicates the state of the Marshall instance. Marshall instances
     * can only change from <CODE>WRITE</CODE> to <CODE>READ</CODE> and
     * this change can only happen once. The idea is that an instance is
     * created, data is marshalled in (<CODE>WRITE</CODE> state) and later
     * data is optionally marshalled out (<CODE>READ</CODE> state).
     */
    protected int state = WRITE;


    /**
     * This returns the size of the current buffer.
     */
    public int getLength()
    {
        if (state == WRITE) return getWriteLength();
        else return getReadLength();
    }


    /**
     * Should return how many characters have been written to the
     * current buffer.
     */
    protected int getWriteLength()
    {
        Log.fatalError(this, "getWriteLength() has not been implemented");

        return 0;
    }


    /**
     * Should return how many characters can be read from this Marshall.
     */
    protected int getReadLength()
    {
        Log.fatalError(this, "getReadLength() has not been implemented");

        return 0;
    }


    /**
     * This should return the data which has been marshalled in the form
     * of a String.
     */
    public String getWriteBuffer()
    {
        return new String(getWriteByteBuffer());
    }

    /**
     * This should return the data which has been marshalled in the form
     * of a String but truncated to the specified length.
     */
    public String getWriteBuffer(int maxLength)
    {
        byte[] bytes = getWriteByteBuffer();
        
        if (bytes.length > maxLength)
            return new String(bytes, 0, maxLength);
        else
            return new String(bytes);
    }

    /**
     * This should return the data which has been marshalled in the form
     * of an array of bytes.
     */
    public byte[] getWriteByteBuffer()
    {
        return getWriteBuffer().getBytes();
    }

    /**
     * This sets the buffer which subsequent unmarshall calls will use.
     */
    public void setReadBuffer(String buffer)
    {
        setReadBuffer(buffer.getBytes());
    }

    public void setReadBuffer(byte[] buffer)
    {
        setReadBuffer(new String(buffer));
    }


    /**
     * This is a utility method for subclasses to ensure that instances
     * are not used illegally with regard to
     * <CODE>READ</CODE>/<CODE>WRITE</CODE> state.
     */
    protected void checkState(int expected)
    {
        if (state != expected)
            throw new RuntimeException("operation illegal in current state");
    }


    // ------------------------------------------------------------------------


    /**
     * This is a utility method for subclasses for calculating padding.
     */
    protected int getPad(int remaining, int required)
    {
        int pad;

        if (remaining < 0) pad = required;
        else
        {
            if (remaining >= required) pad = 0;
            else pad = required - remaining;
        }

        return pad;
    }


    // ------------------------------------------------------------------------


    public abstract Marshall writeChar(String field, int length);


    public abstract Marshall writeVarchar(String field, int length);


    public abstract Marshall writeSmallint(int field);


    public abstract Marshall writeInteger(int field);
    

    public abstract Marshall writePic(double field,
        boolean signed, int length, int fraction);


    public abstract Marshall writeDec(double field, int length, int fraction);


    public abstract Marshall writeDate(java.sql.Date field);


    public abstract Marshall writeTime(java.sql.Time field);


    public abstract Marshall writeTimestamp(java.sql.Timestamp field);


    public abstract Marshall writeBoolean(boolean field);
    
    // ------------------------------------------------------------------------


    public abstract String readChar(int length);


    public abstract String readVarchar(int length);


    public abstract int readSmallint();


    public abstract int readInteger();


    public abstract double readPic(boolean signed, int length, int fraction);


    public abstract double readDec(int length, int fraction);


    public abstract java.sql.Date readDate();


    public abstract java.sql.Time readTime();


    public abstract java.sql.Timestamp readTimestamp();


    public abstract boolean readBoolean();
    
    // ------------------------------------------------------------------------


    private int previous = 0;


    //
    // This function must be called in setReadBuffer() is called.
    //
    protected void setReadBufferLength(int length) { previous = length; }


    /**
     * This method is useful for debugging. Subclasses can override it
     * to log information, etc.
     */
    public void viewStart() { }

    public void viewEnd() { }

    /**
     * This method can optionally be overridden to provide diagnostics.
     */
    public void dumpBuffer()
    {
        Log.fatalError(this, "dumpBuffer() has not been implemented");
    }

    /**
     * This method is provided as a basic tool for subclasses to
     * override <CODE>dumpBuffer()</CODE> above.
     */
    protected void dumpBuffer(String buffer)
    {
        Log.information(this, "buffer length is " + buffer.length());

        char[] src = buffer.toCharArray();

        int len = src.length + ((src.length + 71) / 72) * 2;

        char[] dest = new char[len];

        int pos = 0;

        for (int i = 0; i < src.length; i++)
        {
            // Wrap line every 72 characters.
            if ((i % 72 == 0) && (i != 0))
            {
                dest[pos++] = '\\';
                dest[pos++] = '\n';
            }

            char c = src[i];

            // Replace non-printing characters with period.
            if ((c < 32 && c != '\t' && c != '\n' && c != '\r') || c > 126)
                c = '.';

            dest[pos++] = c;
        }

        dest[pos++] = '\n';

        Log.output(new String(dest, 0, pos));
    }
}
