package com.ibm.ie.reeng.rt.common;


import java.text.SimpleDateFormat;


/**
 * This differs from {@link DefaultMarshall} by marshalling Dates, Times
 * and Timestamps using the ISO character based format.
 */
public class IsoMarshall extends DefaultMarshall
{
    // these are all initialized below.
    private static String ISO_DATE_FORMAT;
    private static int ISO_DATE_FORMAT_LENGTH;
    private static SimpleDateFormat ISO_DATE;

    private static String ISO_TIME_FORMAT;
    private static int ISO_TIME_FORMAT_LENGTH;
    private static SimpleDateFormat ISO_TIME;

    // Note that the java.text.SimpleDateFormat class cannot format sub
    // millisecond precision. Therefore we logically split timestamps
    // into a Datetime portion and a fraction portion.
    private static String DATETIME_FORMAT;
    private static int DATETIME_FORMAT_LENGTH;
    private static SimpleDateFormat DATETIME;
    private static int FRACTION_DIGITS;
    private static int NANOS_PER_FRACTION_UNIT;

    /**
     * It's slightly easier to read by putting all the initialization
     * here. It also reduces the dependency on initialization ordering.
     */
    static
    {
        ISO_DATE_FORMAT = "yyyy-MM-dd";
        ISO_DATE_FORMAT_LENGTH = ISO_DATE_FORMAT.length();
        ISO_DATE = new SimpleDateFormat(ISO_DATE_FORMAT);

        ISO_TIME_FORMAT = "hh:mm:ss";
        ISO_TIME_FORMAT_LENGTH = ISO_TIME_FORMAT.length();
        ISO_TIME = new SimpleDateFormat(ISO_TIME_FORMAT);

        DATETIME_FORMAT = "yyyy-MM-dd hh:mm:ss";
        DATETIME_FORMAT_LENGTH = DATETIME_FORMAT.length();
        DATETIME = new SimpleDateFormat(DATETIME_FORMAT);

        FRACTION_DIGITS = 6;       // ISO compliant
        NANOS_PER_FRACTION_UNIT = 1000;     // i.e. 10^(9 - FRACTION_DIGITS)
    }

    public Marshall writeDate(java.sql.Date field)
    {
        put(ISO_DATE.format(field));
        return this;
    }

    public Marshall writeTime(java.sql.Time field)
    {
        put(ISO_TIME.format(field));
        return this;
    }

    public Marshall writeTimestamp(java.sql.Timestamp field)
    {
        put(DATETIME.format(field));
        put(".");
        String fract
            = Integer.toString(field.getNanos() / NANOS_PER_FRACTION_UNIT);
        put(Util.padLeft(fract, FRACTION_DIGITS, Util.ZERO));
        return this;
    }

    public java.sql.Date readDate()
    {
        try
        {
            String string = get(ISO_DATE_FORMAT_LENGTH);
            return new java.sql.Date(ISO_DATE.parse(string).getTime());
        }
        catch (Exception e)
        {
            Log.fatalException(e);
            return null;    // keep the compiler happy
        }
    }

    public java.sql.Time readTime()
    {
        try
        {
            String string = get(ISO_TIME_FORMAT_LENGTH);
            return new java.sql.Time(ISO_TIME.parse(string).getTime());
        }
        catch (Exception e)
        {
            Log.fatalException(e);
            return null;    // keep the compiler happy
        }
    }

    public java.sql.Timestamp readTimestamp()
    {
        String string = get(DATETIME_FORMAT_LENGTH);
        try
        {
            java.util.Date date = DATETIME.parse(string);

            java.sql.Timestamp ts = new java.sql.Timestamp(date.getTime());

            get(1); // discard the period.
            string = get(FRACTION_DIGITS);
            int nanos = NANOS_PER_FRACTION_UNIT * Integer.parseInt(string);

            ts.setNanos(nanos);

            return ts;
        }
        catch (Exception e)
        {
            Log.fatalException(e);
            return null;    // keep the compiler happy
        }
    }
}
