package com.ibm.ie.reeng.rt.common;

import java.lang.reflect.Array;



/**
 * This class contains implementations of Hps math functions, character
 * functions and the setencoding support function.
 * <P>
 * The math and support function mapping is straightforward; the
 * character function mapping is much less so because the difference
 * between CHAR and VARCHAR is lost in the translation to Java - both
 * are mapped to String.
 * <p>
 * This mapping works better for VARCHARs than for CHAR as their
 * semantics more closely matches that of java.lang.Strings. However the
 * truncation aspect of the Hps character types (e.g. if you assign a
 * longer CHAR or VARCHARs to a shorter) is not supported.
 * <p>
 * The following rule code fragment illustrates some of the situations
 * where the mapping has difficulty:
 * <INDENT>
 * <PRE>
 * dcl
 *   CHAR_STRING    char(4);
 *   VARCHAR_STRING varchar(4);
 * enddcl
 * 
 *  *>
 *  * 1 = strpos(CHAR_STRING, ' ')    - CHAR_STRING is all spaces,
 *  * 0 = strpos(VARCHAR_STRING, ' ') - VARCHAR_STRING is zero length.
 * <*
 * 
 * map '12' to L_CHAR_DATE
 * map '12' to L_VARCHAR_DATE
 * 
 *  *>
 *  * 3 = strpos(CHAR_STRING, ' ')    - CHAR_STRING contains two
 *  *                                   trailing spaces,
 *  * 0 = strpos(VARCHAR_STRING, ' ') - VARCHAR_STRING is length 2.
 * <*
 * 
 * map '1234' to L_CHAR_DATE
 * map '1234' to L_VARCHAR_DATE
 * 
 *  *>
 *  * 0 = strpos(CHAR_STRING, ' ')    - CHAR_STRING is all non-space
 *  *                                   characters
 *  * 0 = strpos(VARCHAR_STRING, ' ') - VARCHAR_STRING is length 4 <*
 * <*
 * </PRE>
 * </INDENT>
 * <P>
 * The implementation of <CODE>strpos</CODE> employs a simple minded
 * heuristic to try and overcome the loss of distinction between CHARs
 * and VARCHARs.  If the searched for string is a single space and the
 * passed in string contains no space it behaves as if that string were
 * of type CHAR and has trailing spaces.  This is less than ideal.
 */
public class Function
{
    // Mathematical functions.


    public static double floor(double x, int power)
    {
        double significant = Math.pow(10, power);

        return Math.floor(x * significant) / significant;
    }

    public static double ceil(double x, int power)
    {
        double significant = Math.pow(10, power);

        return Math.ceil(x * significant) / significant;
    }

    public static double round(double x, int power)
    {
        double significant = Math.pow(10, power);

        return Math.round(x * significant) / significant;
    }

    public static double trunc(double x, int places)
    {
        if (x >= 0) return floor(x, places);
        else return ceil(x, places);
    }

    // ------------------------------------------------------------------------


    // Character functions.

    /**
     * This is used to support the Hps "feature" where you can assign
     * to the underlying length component of a varchar.
     */
    public static String setLength(String s, int len)
    {
        return Util.padRight(s, len, Util.SPACE);
    }


    public static String rtrim(String string)
    {
        return Util.unpadRight(string, Util.SPACE);
    }


    public static String upper(String string)
    {
        return string.toUpperCase();
    }


    public static String lower(String string)
    {
        return string.toLowerCase();
    }


    public static int strlen(String string)
    {
        return rtrim(string).length();
    }


    public static int strpos(String string, String substring)
    {
        int result = string.indexOf(substring);

        if (result == -1)
        {
            // See note above.
            if (substring.equals(" ")) return (string.length() + 1);
            else return 0;
        }
        else return (result + 1);
    }

    public static int charStrpos(String string, String substring)
    {
        int result = string.indexOf(substring);

        if (result == -1)
        {
            // See note above.
            if (substring.equals(" ")) return (string.length() + 1);
            else return 0;
        }
        else return (result + 1);
    }

    public static int verify(String string, String characters)
    {
        for (int i = 0; i < string.length(); i++)
            if (characters.indexOf(string.charAt(i)) == -1) return (i + 1);

        return 0;
    }

    public static int verifyChar(int len, String string, String characters)
    {
        int i = verify(string, characters);

        if (string.length() < len)
            return i == 0 ? string.length() + 1 : i;
        else return i;
    }


    public static String substr(String string, int start, int length)
    {
        start--;
        
        if (start >= string.length() || length < 1) return "";

        if (start < 0) start = 0;

        int end = start + length;

        if (end > string.length()) end = string.length();

        return string.substring(start, end);
    }

    public static String substr(String string, int start)
    {
        return substr(string, start, (string.length() - (start - 1)));
    }


    // ------------------------------------------------------------------------


    // Support functions.


    /**
     * Note: in HPS SETENCODING's return type depends on the type of
     * the values in the set passed to it. In Java this isn't possible,
     * At the moment it so happens that it's only used for sets containing
     * strings.
     */
    public static String setencoding(Set set, String name)
    {
        Object value = set.getValue(name);

        return (value == null) ? "" : value.toString();
    }

 
    private static String chars(char c, int length)
    {
        char[] cs = new char[length];
        for (int i = 0; i < length; i++) cs[i] = c;
        return new String(cs);
    }

    public static String highValues(int length)
    {
        // I'm not sure about this:
        return chars('\uffff',length);
    }

    // ------------------------------------------------------------------------


    // Sundry functions.

    /**
     * Like the built in Java method but doesn't throw an exception.
     */
    public static int parseInt(String s)
    {
        try
        {
            return Integer.parseInt(s);
        }
        catch (NumberFormatException e)
        {
            // not sure what should happen here.
            return 0;
        }
    }

    public static int occurs(Object o)
    {
        if (o == null)
            return 0;
        if (!o.getClass().isArray())
            return 0;
        else
            return Array.getLength(o);
    }

    // support evil view comparison:

    public static int compareView(View v1, View v2)
    {
        return compare(overlay(v1), overlay(v2));
    }

    public static int compareView(View[] v1, View v2)
    {
        return compare(overlay(v1), overlay(v2));
    }

    public static int compareView(View v1, View[] v2)
    {
        return compare(overlay(v1), overlay(v2));
    }

    public static int compareView(View[] v1, View[] v2)
    {
        return compare(overlay(v1), overlay(v2));
    }

    private static OverlayMarshall overlay(View v)
    {
        OverlayMarshall o = new OverlayMarshall();
        v.writeTo(o);
        return o;
    }

    private static OverlayMarshall overlay(View[] v)
    {
        OverlayMarshall o = new OverlayMarshall();
        for (int i = 0; i < v.length; i++)
            v[i].writeTo(o);
        return o;
    }

    private static int compare(OverlayMarshall o1, OverlayMarshall o2)
    {
        byte[] b1 = o1.getWriteByteBuffer();
        byte[] b2 = o2.getWriteByteBuffer();

        int l = Math.min(b1.length, b2.length);
        for (int i = 0; i < l; i++)
        {
            int c = compare(b1[i], b2[i]);
            if (c != 0) return c;
        }

        return b1.length - b2.length;
    }

    // presumably Hps byte comparison is unsigned:
    // there is probably a more optimised way of doing this
    private static int compare(byte b1, byte b2)
    {
        int i1 = (int)b1;
        int i2 = (int)b2;

        int j1 = i1 < 0 ? 256 + i1 : i1;
        int j2 = i2 < 0 ? 256 + i2 : i2;

        return j1 - j2;
    }

    
}
