package com.ibm.ie.reeng.rt.common;


/**
 * This class is provided for debugging purposes. It acts as a wrapper
 * around another {@link Marshall} instance. It simply delegates all
 * methods to the wrapped <code>Marshall</code> after writing details
 * about the call to the log.
 * <P>
 * To debug marshalling, you should probably replace
 * <pre>
 *    server.marshall = <class name>
 * </pre>
 * with
 * <pre>
 *    server.marshall = transmute.runtime.FilterMarshall
 *    filtermarshall.realmarshall = <class name>
 * </pre>
 * in the runtime properties.
 */
public class FilterMarshall extends Marshall
{
    private static final String CHILD_CLASS_KEY
        = "filtermarshall.realmarshall";

    private Marshall child;

    public FilterMarshall(Marshall child) { this.child = child; }

    /**
     * This constructor finds it's child using a system property.
     */
    public FilterMarshall()
    {
        // Get the real Marshall class.
        String marshallClassName
            = Config.instance().getString(CHILD_CLASS_KEY);
        
        if (marshallClassName == null)
        {
            Log.fatalError(this,
                CHILD_CLASS_KEY + " system property is not set.");
            return;
        }

        try
        {
            Class marshallClass = Class.forName(marshallClassName);
            child = (Marshall)marshallClass.newInstance();
        }
        catch (ClassNotFoundException e)
        {
            Log.fatalError(this, "cannot load class " + marshallClassName);
        }
        catch (Exception e)
        {
            Log.fatalError(this, "error creating Marshall class "
                + marshallClassName + ": " + e);
        }
    }

    public int getLength() { return child.getLength(); }

    public String getWriteBuffer() { return child.getWriteBuffer(); }
    public byte[] getWriteByteBuffer() { return child.getWriteByteBuffer(); }

    public void setReadBuffer(String buffer) { child.setReadBuffer(buffer); }
    public void setReadBuffer(byte[] buffer) { child.setReadBuffer(buffer); }

    public void dumpBuffer() { child.dumpBuffer(); }


    public void viewStart()
    {
        Log.information(this, "starting to marshall view");
    }

    public void viewEnd()
    {
        Log.information(this,
            "finished marshalling view. length is " + getLength());
    }

    public Marshall writeChar(String field, int length)
    {
        logChar(field, length);

        child.writeChar(field, length);

        return this;
    }


    public Marshall writeVarchar(String field, int length)
    {
        logVarchar(field, length);

        child.writeVarchar(field, length);

        return this;
    }


    public Marshall writeSmallint(int field)
    {
        logSmallint(field);

        child.writeSmallint(field);

        return this;
    }


    public Marshall writeInteger(int field)
    {
        logInteger(field);

        child.writeInteger(field);

        return this;
    }


    public Marshall writePic(double field,
        boolean signed, int length, int fraction)
    {
        logPic(field, signed, length, fraction);

        child.writePic(field, signed, length, fraction);

        return this;
    }


    public Marshall writeDec(double field, int length, int fraction)
    {
        logDec(field, length, fraction);

        child.writeDec(field, length, fraction);

        return this;
    }


    public Marshall writeDate(java.sql.Date field)
    {
        logDate(field);

        child.writeDate(field);

        return this;
    }


    public Marshall writeTime(java.sql.Time field)
    {
        logTime(field);

        child.writeTime(field);

        return this;
    }


    public Marshall writeTimestamp(java.sql.Timestamp field)
    {
        logTimestamp(field);

        child.writeTimestamp(field);

        return this;
    }

    public Marshall writeBoolean(boolean field)
    {
        logBoolean(field);

        child.writeBoolean(field);

        return this;
    }

    

    public String readChar(int length)
    {
        String field = child.readChar(length);

        logChar(field, length);

        return field;
    }


    public String readVarchar(int length)
    {
        String field = child.readVarchar(length);

        logVarchar(field, length);

        return field;
    }


    public int readSmallint()
    {
        int field = child.readSmallint();

        logSmallint(field);

        return field;
    }


    public int readInteger()
    {
        int field = child.readInteger();

        logInteger(field);

        return field;
    }


    public double readPic(boolean signed, int length, int fraction)
    {
        double field = child.readPic(signed, length, fraction);

        logPic(field, signed, length, fraction);

        return field;
    }


    public double readDec(int length, int fraction)
    {
        double field = child.readDec(length, fraction);

        logDec(field, length, fraction);

        return field;
    }


    public java.sql.Date readDate()
    {
        java.sql.Date field = child.readDate();

        logDate(field);

        return field;
    }


    public java.sql.Time readTime()
    {
        java.sql.Time field = child.readTime();

        logTime(field);

        return field;
    }


    public java.sql.Timestamp readTimestamp()
    {
        java.sql.Timestamp field = child.readTimestamp();

        logTimestamp(field);

        return field;
    }

    public boolean readBoolean()
    {
        boolean field = child.readBoolean();

        logBoolean(field);

        return field;
    }
    

    // ------------------------------------------------------------------------


    private void logChar(String field, int length)
    {
        Log.information(this,
            "CHAR(" + length + ") \"" + field + "\"");
    }


    private void logVarchar(String field, int length)
    {
        Log.information(this,
            "VARCHAR(" + length + ") \"" + field + "\"");
    }


    private void logSmallint(int field)
    {
        Log.information(this, "SMALLINT " + field);
    }


    private void logInteger(int field)
    {
        Log.information(this, "INTEGER " + field);
    }
    

    private void logPic(double field,
        boolean signed, int length, int fraction)
    {
        String picture = "";

        if (signed)
        {
            picture = "S";
            length--;
        }

        int integer = length - fraction;

        if (integer > 0) picture += "9(" + integer + ")";

        if (fraction > 0) picture += "V9(" + fraction + ")";

        Log.information(this, "PIC '" + picture + "' " + field);
    }


    private void logDec(double field, int length, int fraction)
    {
        String string = "";

        if (fraction > 0) string = ", " + fraction;

        Log.information(this, "DEC(" + length + string +
            ") " + field);
    }


    private void logDate(java.sql.Date field)
    {
        Log.information(this, "DATE " + field);
    }


    private void logTime(java.sql.Time field)
    {
        Log.information(this, "TIME " + field);
    }


    private void logTimestamp(java.sql.Timestamp field)
    {
        Log.information(this, "TIMESTAMP " + field);
    }

    private void logBoolean(boolean field)
    {
        Log.information(this, "BOOLEAN " + field);
    }
}
