package com.ibm.ie.reeng.rt.common;

/**
 * This class supports marshalling to a BLOB or octet stream. It
 * emulates the in memory layout of Hps data. It is used as a base class
 * for {@link OverlayMarshall} but could also potentially be used as the
 * basis of a more space efficient marshalling for client/host
 * communication.
 * <p>
 * See {@link Marshall} for a general discussion on
 * marhalling/unmarshalling.
 */
public abstract class ByteMarshall extends Marshall
{
    public boolean readBoolean()
    {
        throw new RuntimeException("boolean marshalling not yet supported");
    }
    
    public Marshall writeBoolean(boolean field)
    {
        throw new RuntimeException("boolean marshalling not yet supported");
    }
    
    private final static int BLOCK = 32 * 1024; // 32K

    protected byte[] buffer = new byte[BLOCK];
    protected int length = 0;

    /**
     * Returns the current BLOB contents.
     * This method can only be used when this class is in the
     * <code>WRITE</code> state.
     */
    public byte[] getWriteByteBuffer()
    {
        checkState(WRITE);

        if (buffer.length != length) resize(length);

        return buffer;
    }


    /**
     * Appends <code>bytes</code> to the byte buffer contents. The
     * buffer is extended if necessary.
     * <p>
     * This method can only be used when this class is in the
     * <code>WRITE</code> state.
     */
    protected void put(byte[] element)
    {
        checkState(WRITE);

        int required = length + element.length;

        if (buffer.length < required) resize(required * 2);

        System.arraycopy(element, 0, buffer, length, element.length);

        length += element.length;
    }


    /**
     * Adds a single <code>byte</code> to the current BLOB contents.
     * <p>
     * This method can only be used when this class is in the
     * <code>WRITE</code> state.
     */
    protected void put(byte b)
    {
        byte[] element = { b };

        put(element);
    }


    /**
     * Resizes the internal byte buffer. Note that the current contents
     * may be truncated.
     */
    protected void resize(int newLength)
    {
        byte[] original = buffer;

        buffer = new byte[newLength];
        System.arraycopy(original, 0, buffer, 0, length);
    }


    /**
     * Returns the number of bytes used in the internal byte buffer.
     */
    protected int getWriteLength() { return length; }


    // ------------------------------------------------------------------------


    protected int position = 0;


    /**
     * Switches state from <code>WRITE</code> to <code>READ</code>.
     * Subsequent <code>readXXX</code> methods use the supplied
     * <code>byte</code> buffer.
     */
    public void setReadBuffer(byte[] buffer)
    {
        checkState(WRITE);

        this.buffer = buffer;

        setReadBufferLength(buffer.length);

        state = READ;
    }


    /**
     * Returns <code>elementLength</code> <code>byte</code>s from the
     * current position in the byte buffer. The current position is
     * advanced appropriately.
     * <p>
     * This method is used directly or indirectly by all
     * <code>readXXX</code> methods to fetch bytes from the internal
     * buffer.
     * <p>
     * Note this method automatically switches the Marshall state from
     * <code>WRITE</code> to <code>READ</code> if necessary.
     */
    protected byte[] get(int elementLength)
    {
        if (state == WRITE) setReadBuffer(getWriteByteBuffer());

        byte[] element = new byte[elementLength];

        int pad = getPad(buffer.length - position, elementLength);
        int count = elementLength - pad;

        if (count > 0)
            System.arraycopy(buffer, position, element, 0, count);

        for (int i = 0; i < pad; i++)
            element[count + i] = (byte)' ';

        position += elementLength;

        return element;
    }

    /** Convienience method for {@link #get(int)}.  */
    protected byte get()
    {
        byte[] element = get(1);

        return element[0];
    }

    /**
     * Returns the number of bytes left in the internal byte buffer.
     */
    protected int getReadLength() { return (buffer.length - position); }


    // ------------------------------------------------------------------------


    /**
     * Dumps the contents of the internal byte buffer. This is useful
     * for debugging and diagnostics.
     */
    public void dumpBuffer() { dumpBuffer(new String(buffer)); }
}
