package com.ibm.ie.reeng.rt.client.component;

import com.ibm.ie.reeng.rt.client.SystemComponent;
import com.ibm.ie.reeng.rt.common.Log;
import com.ibm.ie.reeng.rt.client.TransmuteDialog;
import com.ibm.ie.reeng.rt.client.AlteredField;
import com.ibm.ie.reeng.rt.common.set.ReturnCodes;
import com.ibm.ie.reeng.rt.common.view.GetAlteredFieldI;
import com.ibm.ie.reeng.rt.common.view.GetAlteredFieldO;

import java.util.ArrayList;

/**
 * This system component determines the screen fields the end
 * user has changed. 
 * It can be used iteratively to find all the altered screen
 * fields or called once to check if a specific field has been altered. 
 * If no view name and no field name are specified in the input view, it looks 
 * for all changed fields in every view. If a field name is specified but a 
 * view name is not, the component searches for the specified field in any view. 
 * If a view name is specified but a field name is not, the component searches 
 * for any changed field within the specified view.
 * <p>
 * The FieldOccur parameter directs the component to return the first
 * changed field found, the next changed field, or a field with a particular 
 * occurrence number in a TransmuteTable. The component returns the view name, field name,
 * and field occurrence of the changed field to the calling rule when the
 * GetAlteredField component is successful. When no more altered fields
 * are found, the return code is set to failure.
 * <p>
 * The scope of the altered field search includes text fields, combo-boxes, tables
 * check boxes, and radio buttons.
 * <p>
 * An iterative call to this component loops through all altered fields unless
 * you clear the altered field Vector of the TransmuteDialog using 
 * {@link RestoreAlteredFields}
 * <p>
 * Assumptions:
 * <ul>
 * <li>each changed cell in a table is a separate return if using the iteration-type call 
 * (i.e. if fieldOccur=0 in input view)
 * <li> calling this component with fieldOccur=0 will not take account of a previous call
 * with fieldOccur=-1, i.e. it may return the same component
 * <li>if you start making a set of iterative calls (i.e. with fieldOccur = 0) and do not
 * complete the full set, or change input parameters from one call to the next, you can
 * expect weird behaviour.
 * </ul>
 */
public class GetAlteredField extends SystemComponent
{
    private String viewLongName;
    private String fieldLongName;
    private int fieldOccur;
    private AlteredField[] alteredFieldArray = new AlteredField[0];
    private int index = 0;
    
    public GetAlteredField() 
    {}

    /**
     * Input view.
     */
    public GetAlteredFieldI in = new GetAlteredFieldI();

    public void clearInputView() { in.clear(); }

    /**
     * Output view.
     */
    public GetAlteredFieldO out = new GetAlteredFieldO();

    protected void clearOutputView() { out.clear(); }

    protected void execute()
    {
        Log.information(this, "in system.component.GetAlteredField.execute()");
        //read inputs
        viewLongName = in.viewLongName;        
        fieldLongName = in.fieldLongName;        
        fieldOccur = in.fieldOccur;      
        
        //set up default output (==failure, which means no altered fields were found)
        out.returnCode = ReturnCodes.FAILURE;
        out.viewLongName = "";
        out.fieldLongName = "";
        out.fieldOccur = 0;

        AlteredField matchField;
        TransmuteDialog dialog = TransmuteDialog.getCurrentDialog();

        if(dialog!=null)
        {
            if(index==0)  
            //means that this call is not recognised as being a subsequent call 
            //which is part of an iterative request
            {
                ArrayList changedFields = dialog.getAlteredFields();
                alteredFieldArray = new AlteredField[changedFields.size()];
                for(int i=0; i<changedFields.size(); i++)
                	alteredFieldArray[i] = (AlteredField)changedFields.get(i);
            }            
        
            matchField = nextMatch(); 
            if(matchField!=null) 
                writeToOutput(matchField);
        
            if(fieldOccur!=0 || matchField==null)    
                index=0;    //reset index so that next call will be independent of this one
                            //otherwise index marks starting point for next iterative "get"
        }
    }

    private void writeToOutput(AlteredField matchField)
    {
        out.returnCode = ReturnCodes.SUCCESS;
        out.viewLongName = matchField.view;
        out.fieldLongName = matchField.field;
        out.fieldOccur = matchField.occur;                            
    }
    
    /**
     * get next member of alteredFieldArray which matches input parameters
     */
    private AlteredField nextMatch()
    {
        AlteredField match = null;
        boolean found=false;
        while(!found && (index<alteredFieldArray.length))
        {
            if (alteredFieldArray[index].match(viewLongName, fieldLongName, fieldOccur))
            {
                match = alteredFieldArray[index];
                found=true;
            }
            index++;
        }
        return match;
    }
}