package com.ibm.ie.reeng.rt.client;

import com.ibm.ie.reeng.rt.common.Log;

import java.util.ArrayList;
import javax.swing.SwingConstants;
import javax.swing.table.AbstractTableModel;


/**
 * The model used by {@link TransmuteTable} instances.
 */
public class TransmuteTableModel extends AbstractTableModel
            implements java.io.Serializable
{
    private TransmuteTable parent;
    private ArrayList columns = new ArrayList();


    public TransmuteTableModel(TransmuteTable parent)
    {
        this.parent = parent;
    }


    public boolean isCellEditable(int row, int column)
    {
        return getColumn(column).isEditable();
    }


    public void addColumn(TransmuteColumn column)
    {
        columns.add(column);
    }

    public ArrayList getColumns()
    {
        return columns;
    }
    
    public TransmuteColumn getColumn(int i)
    {
        return (TransmuteColumn)columns.get(i);
    }


    public String getColumnName(int i)
    {
        return getColumn(i).toString();
    }


    public int getColumnCount() { return columns.size(); }


    public int findColumnByIdentifier(String match)
    {
        for (int i = 0; i < getColumnCount(); i++)
        {
            if (getColumn(i).getIdentifier().equals(match)) return i;
        }

        Log.error(this, "could not find column " + match);

        return -1;
    }


    // ------------------------------------------------------------------------

    public TransmuteColumn getFirstImmediateReturnColumn()
    {
        for (int i = 0; i < getColumnCount(); i++)
        {
            if (getColumn(i).isImmediateReturn()) return getColumn(i);
        }

        return null;
    }

    // ------------------------------------------------------------------------


    public void resetState()
    {
        viewStartPos = -1;
        viewEndPos = -1;

        backBuffer = 1;
        virtualRowCount = -1;
        offset = 0;
        eventsEnabled = true;

        for (int i = 0; i < getColumnCount(); i++)
            getColumn(i).resetState();
    }


    // ------------------------------------------------------------------------


    private int viewStartPos = -1;
    private int viewEndPos = -1;


    public int getViewStartPos()
    {
        if (viewStartPos == -1) viewStartPos = 0;

        return viewStartPos;
    }


    public int getViewEndPos()
    {
        if (viewEndPos == -1) viewEndPos = getRealRowCount() - 1;

        return viewEndPos;
    }


    public int getVirtualStartPos()
    {
        return offset + getViewStartPos();
    }


    public int getVirtualEndPos()
    {
        return offset + getViewEndPos();
    }


    // ------------------------------------------------------------------------


    private int backBuffer = 1;
    private int virtualRowCount = -1;
    private int offset = 0;
    private boolean eventsEnabled = true;


    public void enableEvents(boolean b) { eventsEnabled = b; }


    public int[] getAvailableRows()
    {
        return new int[]{ offset, (offset + getRealRowCount() - 1) };
    }


    //
    // getRowCount() is required by the TableModel interface, however
    // getVirtualRowCount() should be used instead in any internal code
    // to clearly distinguish whether the real or virtual row count is
    // being used.
    //
    public int getRowCount() { return getVirtualRowCount(); }


    private int getVirtualRowCount()
    {
        if (virtualRowCount != -1) return virtualRowCount;
        else return getRealRowCount();
    }


    public int getRealRowCount()
    {
        int count = Integer.MAX_VALUE;

        for (int i = 0; i < columns.size(); i++)
            count = Math.min(count, getColumn(i).getRowCount());

        return count;
    }


    public void setVirtualRowCount(int count)
    {
        int previous = getVirtualRowCount();

        if (previous == count) return;

        virtualRowCount = count;

        if (virtualRowCount > previous)
            fireTableRowsInserted(previous, (virtualRowCount - 1));
        else fireTableRowsDeleted(virtualRowCount, (previous - 1));
    }


    public void setBackBuffer(int backBuffer) { this.backBuffer = backBuffer; }


    public void visibleRangeChanged(boolean down, int top, int bottom)
    {
        int count = getRealRowCount();
        int newOffset = -1;

        if (top < offset) // Scrolled up exposing new row(s) at top.
        {
            newOffset = bottom - (count - 1);

            // Offset is changing - do basic sanity check.
            checkCapacity();

            // Keep X non-visible row below the bottom.
            newOffset += backBuffer;

            if (newOffset < 0) newOffset = 0;
        }
        else
        {
            int max = offset + (count - 1);

            if (bottom > max) // Scrolled down exposing new row(s) at bottom.
            {
                newOffset = top;

                // Offset is changing - do basic sanity check.
                checkCapacity();

                // Keep X non-visible row above the top;
                newOffset -= backBuffer;
            }
        }

        if (newOffset != -1)
        {
            if (newOffset < 0) newOffset = 0;
            else
            {
                int bottomBoundary = getVirtualRowCount() - count;

                if (newOffset > bottomBoundary) newOffset = bottomBoundary;
            }

            // If events are disabled just update values and bail out.
            if (!eventsEnabled)
            {
                offset = newOffset;
                viewStartPos = -1;
                viewEndPos = -1;
                return;
            }

            // If all data is okay update the offset...
            if (storeData() == DataHandler.VALID)
            {
                int diff = Math.abs(newOffset - offset);

                if (diff >= count)
                {
                    viewStartPos = 0;
                    viewEndPos = count - 1;
                }
                else
                {
                    // Copy any data that's still valid to its new position.

                    if (down)
                    {
                        for (int i = diff; i < count; i++)
                            copyRow(i, (i - diff));

                        viewStartPos = count - diff;
                        viewEndPos = count - 1;
                    }
                    else
                    {
                        for (int i = (count - 1); i >= diff; i--)
                            copyRow((i - diff), i);

                        viewStartPos = 0;
                        viewEndPos = diff - 1;
                    }
                }

                // Clear the rows that need to be updated externally.
                for (int i = viewStartPos; i <= viewEndPos; i++) clearRow(i);

                offset = newOffset;

                fireTableDataChanged();
            }

            // Rather than fireTableDataChanged() the normal function
            // TransmuteDialog.updateSharedDatalinks() should be
            // called but it does not work with TransmuteColumn.

            // Whether the data is valid or not generate an event.
            // If the data is invalid the user will be informed.

            String qualifier = down ? TransmuteDialog.TABLE_EVENT_DOWN :
                TransmuteDialog.TABLE_EVENT_UP;

            TransmuteDialog.getParentDialog(parent).
                dispatchTableOutRangeEvent(parent, qualifier);
        }
    }

    public int getElevatorPosition() { return offset; }

    public void setElevatorPosition(int offset)
    {
        this.offset = offset;
        fireTableDataChanged();
    }

    public void checkCapacity()
    {
        int realCount = getRealRowCount();
        int visibleCount = parent.getVisibleRowCount();

        if ((visibleCount + backBuffer) > realCount)
        {
            if (visibleCount > realCount)
                Log.fatalError(this, "tables linked view is too small");
            else
            {
                Log.error(this, "backBuffer size is too large");

                // This is okay - just cut backBuffer down to size.
                backBuffer = realCount - visibleCount;
            }
        }
    }


    private void copyRow(int sourceRow, int destRow)
    {
        for (int i = 0; i < getColumnCount(); i++)
        {
            TransmuteColumn column = getColumn(i);

            column.set(column.get(sourceRow), destRow);
        }
    }


    private void clearRow(int row)
    {
        for (int i = 0; i < getColumnCount(); i++) getColumn(i).clear(row);
    }


    public Object getValueAt(int row, int col)
    {
        TransmuteColumn column = getColumn(col);
        int count = column.getRowCount();

        if (row < offset || row >= (offset + count)) return "";
        else return getColumn(col).getString(row - offset);
    }


    public void setValueAt(Object value, int row, int col)
    {
        TransmuteColumn column = getColumn(col);
        int count = column.getRowCount();

        // This should never happen...
        if (row < offset || row >= (offset + count)) return;

        column.setString((String)value, (row - offset));
        fireTableCellUpdated(row, col);
        
        //calls method on column to specify AlteredField.  
        //Note that HPS row numbering begins with 1, while java indexing begins at 0
        column.setAlteredField(row+1);        
    }


    public int[] getFirstErrorCell()
    {
        for (int i = 0; i < getColumnCount(); i++)
        {
            int row = getColumn(i).getFirstErrorRowIndex();

            if (row != -1) return new int[]{ (row + offset), i };
        }

        return null;
    }


    public boolean isErrorCell(int row, int col)
    {
        return getColumn(col).isErrorRow(row - offset);
    }


    public int storeData()
    {
        if (parent.isEditing()) parent.getCellEditor().stopCellEditing();

        if (getFirstErrorCell() != null) return DataHandler.INVALID;

        return DataHandler.VALID;
    }
}
