package com.ibm.ie.reeng.rt.client;

import javax.swing.JComboBox;
import javax.swing.AbstractListModel;
import javax.swing.ComboBoxModel;
import javax.swing.ComboBoxEditor;
import javax.swing.plaf.basic.BasicComboBoxEditor;
import javax.swing.border.Border;
import java.awt.Color;
import java.awt.SystemColor;
import java.awt.event.FocusListener;

import com.ibm.ie.reeng.rt.common.Config;

/**
 * This extends {@link JComboBox} to provide Hps combo box
 * functionality.
 * <P>
 * Most of the behavior and visual appearance of a combo box are really
 * determined by the combo box editor which is usually a JTextField.
 * However calls like addFocusListener, setForeground and setBackround
 * do not correctly delegate to this control due to known bugs.
 * See the Java developers' bug parade for numerous combo box bugs such
 * as 4243641.
 */
public class TransmuteComboBox extends JComboBox
    implements Identifier, DataHandler.Interface, TextDataHandler.Interface, java.io.Serializable
{
    private ClientData clientData;
    private DataModel model = new DataModel();
    private TextDataHandler handler;
    
    // flag to indicate whether to allow Null selection
    // true by default
 	private static boolean allowNullSelection = true;

	static
	{
		allowNullSelection = Config.instance().getBoolean("tool.java.widget.combobox.allowNullSelection");    	
	}

    public TransmuteComboBox()
    {
        setEditor();

        // Due to a bug in the implementation of addFocusListener(),
        // see below, the data handler must be created after
        // setEditor() is called.
        handler = new TextDataHandler(this);        
    }
    

    public DataHandler getDataHandler() { return handler; }


    public void setMandatory(boolean mandatory)
    {
        getDataHandler().setMandatory(mandatory);
    }

    public boolean getMandatory()
    {
        return getDataHandler().getMandatory();
    }

    public void setImmediateReturn(boolean immediateReturn)
    {
        getDataHandler().setImmediateReturn(immediateReturn);
    }

    public boolean getImmediateReturn()
    {
        return getDataHandler().getImmediateReturn();
    }

    public void setClientData(ClientData clientData)
    {
        this.clientData = clientData;

        setModel(model);

        handler.setClientData(clientData);
    }


    public String getText() 
    {
        if (getSelectedItem()==null)
            return "";
        else
            return getSelectedItem().toString(); 
    }


    public void setText(String text)
    {
        setSelectedItem(text);

        model.fireContentsChanged();
        
        //specifies new AlteredField for TransmuteDialog
        TransmuteDialog tdParent = TransmuteDialog.getParentDialog(this);
        if (tdParent != null)
        	tdParent.setAlteredField(new AlteredField(handler.getPrimaryView(), handler.getPrimaryField()));
    }


    // ------------------------------------------------------------------------


    /**
     * In HPS you can set the height of a combo box to any
     * arbitrarily large value but it just ignores that value
     * and uses the normal height. The following does the same.
     */
    public void setBounds(int x, int y, int width, int height)
    {
        height = getPreferredSize().height;

        super.setBounds(x, y, width, height);
    }


    public void setMaxLength(int length)
    {
// Calls to this should only be generated if readonly is off but...
    }


    // ------------------------------------------------------------------------


    /**
     * Note that the data model contains one more element than the
     * ClientData. This is because Hps allows "null" or empty selection
     * from the possibilities.
     * GJ 20040303 the "allowNullSelection" is now a property in the runtime.properties
     */
    class DataModel extends AbstractListModel implements ComboBoxModel,
               java.io.Serializable
    {
        Object selected = "";
                

        public Object getSelectedItem() { return selected; }

        public void setSelectedItem(Object selected)
        {
            this.selected = selected; 
            fireContentsChanged(this, -1, -1);
        }

        public int getSize()
        {
			if(allowNullSelection)			
				return (clientData != null) ? (clientData.getSize() + 1): 1;						        
            return (clientData != null) ? (clientData.getSize()): 0;
        }

        public Object getElementAt(int i)
        {
        	if(allowNullSelection)
        	{
				if (clientData != null)
					return (i == 0) ? "" : clientData.getString(i - 1);
				else return (i == 0) ? selected : null;        		
        	}
        	else
        	{
				if (clientData != null)
					return clientData.getString(i);
        	}
            return null;
        }

        public void fireContentsChanged()
        {
            fireContentsChanged(this, 0, (getSize() - 1));
        }
    }
    


    // ------------------------------------------------------------------------


    private String identifier;

    public String getIdentifier() { return identifier; }

    public void setIdentifier(String identifier)
    {
        this.identifier = identifier;
    }


    // ------------------------------------------------------------------------



    /**
     * This is used to workaround a JDK bug.
     * See {@link JTextFieldBugFix} for more information.
     */
    static class BasicComboBoxEditorBugFix extends BasicComboBoxEditor
            implements java.io.Serializable
    {
        public BasicComboBoxEditorBugFix()
        {
            editor = new JTextFieldBugFix(9)
            {
                public void setBorder(Border b) { }
            };
            editor.addFocusListener(this);
        }
    }


    private void setEditor()
    {
        super.setEditor(new BasicComboBoxEditorBugFix());
    }


    public void setEditor(ComboBoxEditor ignore)
    {
        setEditor();
    }


    // ------------------------------------------------------------------------


    public void addFocusListener(FocusListener listener)
    {
        if (getEditor() == null) super.addFocusListener(listener);
        else getEditor().getEditorComponent().addFocusListener(listener);
    }


    public Color getForeground()
    {
        if (getEditor() == null) return super.getForeground();
        else return getEditor().getEditorComponent().getForeground();
    }


    public void setForeground(Color c)
    {
        if (getEditor() == null) super.setForeground(c);
        else getEditor().getEditorComponent().setForeground(c);
    }


    public Color getBackground()
    {
        if (getEditor() == null) return super.getBackground();
        else return getEditor().getEditorComponent().getBackground();
    }


    public void setBackground(Color c)
    {
        if (getEditor() == null) super.setBackground(c);
        else getEditor().getEditorComponent().setBackground(c);
    }
}
