package com.ibm.ie.reeng.rt.client;

import com.ibm.ie.reeng.rt.common.Log;
import com.ibm.ie.reeng.rt.common.LocalRule;

import com.ibm.ie.reeng.rt.client.State;

import java.io.*;
import java.util.*;


/**
 * The base class for all generated Java client rule classes.
 */
public abstract class Rule extends LocalRule
{
    public static class UseMode implements java.io.Serializable
    {
        public static final UseMode NORMAL=new UseMode("NORMAL");
        public static final UseMode NEST=new UseMode("NEST");
        public static final UseMode DETACH=new UseMode("DETACH");
        public static final UseMode DETACHED=new UseMode("DETACHED");
        
        private String type;
        
        private UseMode(String s)
        {
            type=s;
        }

        public String toString()
        {
            return "UseMode:"+type;
        }
    }

    protected final static boolean NOWAIT = true;

    /** Use Rule .. Detach Instance .. not currently supported.  Therefore,
     * only allow one detached child rule.  We would use a HashMap here
     * to identify instances by name otherwise.
     */
    private Rule detachedChild;

    public synchronized void setDetachedChild(Rule detachedChild)
    {
        this.detachedChild=detachedChild;
    }

    public synchronized Rule getDetachedChild()
    {
        return detachedChild;
    }

    /**
     * eventDest records where user events are to be sent when a detached
     * rule uses HPS_EVENT_POST_TO_PARENT.  Note that this is not always the
     * same as detachedParent.
     */
    private Rule eventDest;

    private boolean interceptingEvents=false;

    /**
     * Note: There seems to be no way of switching off interceptingEvents
     */
    
    public void setInterceptingEvents()
    {
System.out.println("Now intercepting events");        
        interceptingEvents=true;
    }

    /**
     * When a child sends an event to it's parent the event destination is:
     * The root rule of the parent (or the parent, if it is the root rule)
     * unless HPS_INTERCEPT_EVENTS has been called by the parent.
     * If this is the case, then the destination is the parent, even if it
     * is not the root rule (Got it?).
     */
    public Rule getEventDest()
    {
        if (interceptingEvents)
            return this;
        else
            return eventDest;
    }

    private Rule detachedParent;

    /** 
     * processTopRule is where events end up when they are posted from a
     * detached child to its parent.  We use threads, not processes, but
     * model the behaviour (where each 'process' has a top level rule).
     */
    private Rule processTopRule;

    /**
     * Subclasses must implement one of these to provide the rule logic.
     * Since subclasses only want to implement one of these
     * we'll provide empty implementations here (rather than specify 
     * abstract methods).
     */

    protected void execute(State ctx) { Log.fatalError(this, "execute(State) not defined"); }
    protected void execute() { Log.fatalError(this, "excecute() not defined!"); }

    private boolean createRules = true;

    public void use(UseMode mode, State ctx) { use(mode, ctx, null); }
    public void use(State ctx) { use(UseMode.NORMAL, ctx, null); }
    public void use(UseMode mode) { use(mode, null, null); }
    public void use() { use(UseMode.NORMAL, null, null); }


    /**
     * This is the public interface to the rule. Subclasses implement
     * {@link #execute} but users call this method which creates child
     * rules (see {@link #createChildRules}) if necessary, clears the
     * local variables, the output view and the input views of child
     * rules.
     */
    public void use(UseMode mode, State ctx, Rule callingRule)
        throws IllegalThreadStateException
    {
        final State _ctx=ctx;
        
        if (mode==UseMode.NEST) Log.information(this, "entering nested");
        else if (mode==UseMode.DETACH) Log.information(this, "entering detached");
        else Log.information(this, "entering");

        if (mode==UseMode.DETACH)
        {
            if (callingRule.getDetachedChild()!=null)
            {
                Log.warning(this, "Cannot detach a second child!");
                return;
            }
            
            final Rule detachedRule=deepCopy();

            callingRule.setDetachedChild(detachedRule);
            if (callingRule.processTopRule == null)
                detachedRule.eventDest=callingRule;
            else
                detachedRule.eventDest=callingRule.processTopRule;

            detachedRule.detachedParent=callingRule;
                
            new Thread(new Runnable()
                        {
                            public void run()
                            {
                                detachedRule.use(UseMode.DETACHED, _ctx);
                            }
                        }
                        ).start();
            return;

        }

        if (createRules)
        {
            createChildRules();
            createRules = false;
        }

        // Clear relevant views and local variables.
        clearLocalVariables();
        clearOutputView();
        clearChildInputViews();

        initializeWindow(mode);

        if (_ctx==null)
            execute();
        else
            execute(_ctx);

        finalizeWindow();

        // The following could be unsafe in situations where the child
        // will outsurvive the parent.
        // It's possible, though very unlikely, that the detachedParent
        // could cease to exist in the fraction of time it takes to go
        // from checking it and calling the method
        if (detachedParent!=null)
            detachedParent.setDetachedChild(null);

        Log.information(this, "exiting");
    }

    /**
     * Used when we need to maintain processTopRule and
     * detachedChild rule relationships.
     * Note: A side effect is to cause this form
     * to be used for calling components.  This is
     * also required when components need to know their
     * calling rule.
     */
    public void use(Rule rule)
    {
        use(rule, UseMode.NORMAL);
    }
    
    public void use(Rule rule, UseMode mode)
    {
        if (mode==UseMode.DETACH)
            rule.use(mode, null, this);
        else
        {
            if (processTopRule==null)
                rule.processTopRule=this;
            else
                rule.processTopRule=processTopRule;
            rule.use(mode, null);
        }
    }

    public void use(Component component)
    {
        component.setCallingRule(this);
        component.use();
    }

    /** Initializes the rule's window if it has one. */
    protected void initializeWindow(UseMode mode) { }

    /** Cleans up the rule's window if it has one. */
    protected void finalizeWindow() { }

    public Rule deepCopy()
    {
        try
        {
            ByteArrayOutputStream baos=new ByteArrayOutputStream(1000);
            ObjectOutputStream oos=new ObjectOutputStream(baos);
            oos.writeObject(this);
            byte buf[]=baos.toByteArray();
            oos.close();

            ByteArrayInputStream bais=new ByteArrayInputStream(buf);
            ObjectInputStream ois=new ObjectInputStream(bais);
            Rule retObj=(Rule)ois.readObject();
            ois.close();

            return retObj;
        }
        catch (Exception e)
        {
            Log.exception(e);
            return null;
        }
    }

    public void converse(TransmuteDialog dialog)
    {
        converse(dialog, false);
    }

    public void converse(TransmuteDialog dialog, boolean nowait)
    {
        dialog.setCallingRule(this);
        dialog.converse(nowait);
    }


    public static class Event
    {
        public Event(String eventName, String eventView, String eventParam, String eventSource)
        {
            this.eventName=eventName;
            this.eventView=eventView;
            this.eventParam=eventParam;
            this.eventSource=eventSource;
        }
        
        private String eventName;
        private String eventView;
        private String eventParam;
        private String eventSource;

        public String getEventName()
        {
            return eventName;
        }

        public String getEventView()
        {
            return eventView;
        }

        public String getEventParam()
        {
            return eventParam;
        }
        
        public String getEventSource()
        {
            return eventSource;
        }
    }

    private ArrayList eventQueue;

    private TransmuteDialog conversingWindow;

    public synchronized void setConversingWindow(TransmuteDialog window)
    {
        conversingWindow=window;
    }

    public synchronized void notifyConversingWindow()
    {
        if (conversingWindow!=null)
        {
            conversingWindow.notifyEvent();
        }
    }

    public synchronized void postEvent(Event event)
    {
        if (eventQueue==null)
            eventQueue=new ArrayList();

        eventQueue.add(event);

        // If this rule is currently conversing a window, we
        // need to notify the converse
        notifyConversingWindow();
    }

    public synchronized Event getEvent()
    {
        if (eventQueue==null || eventQueue.isEmpty())
            return null;
        else
        {
            return (Event)eventQueue.remove(0);
        }
    }

    public synchronized boolean hasEvent()
    {
        return eventQueue!=null && !eventQueue.isEmpty();
    }

    /**
     * Allows arbitrary rules to be executed from the command line.
     */
    public static void main(String[] args)
    {
        try
        {
            if (args.length != 1 && args.length != 2)
            {
                System.err.println("Usage: java "
                    + "com.ibm.ie.reeng.rt.client.ClientRule"
                    + " <full rule class name>");
                System.exit(-1);
            }
            Log.information("ClientRule", "starting..");

            String startRule = args[0];

            ClientRule root = (ClientRule)Class.forName(startRule).newInstance();

//            State state=new State();

//            root.use(state);

            root.use();

            Log.information("ClientRule", startRule
                + " exiting normally...");

            System.exit(0);
        }
        catch (Exception e)
        {
            Log.fatalException(e);
        }
    }
}
