package com.ibm.ie.reeng.rt.client;

import com.ibm.ie.reeng.rt.common.Log;
import com.ibm.ie.reeng.rt.common.Config;

import java.io.File;


/**
 * This class supports the help system. It is requred by the
 * SetHelpFileName, HpsSetHelpTopic and ShowHelpTopic system components.
 * It uses JNI to interact with the win32 help API. Thus Help.cpp needs
 * to be compiled to a dll before this class can be used.
 * <P>
 * Native Hps help is not supported.
 */
public class Help
{
    /**
     * The system property key for the style of help
     * (&quot;help.style;&quot;). Currently supported settings include
     * &quot;winhelp&quot;, &quot;htmlhelp&quot; and &quot;none&quot;.
     */
    public static final String HELP_SYTLE_KEY = "help.style";

    /**
     * The system property key for the help files directory
     * (&quot;help.dir;&quot;). 
     */
    public static final String HELP_DIR_KEY = "help.dir";

    /**
     * The system property key for the default help file
     * (&quot;help.default.file;&quot;). If the rebased client uses a
     * single help file then this property can be set appropriately and
     * all uses of {@link transmute.runtime.system.component.SetHelpFileName}
     * can be removed from the code.
     */
    public static final String HELP_DEFAULT_FILE = "help.default.file";

    /** Standard windows help. Uses &quot;.HLP&quot; files. */ 
    public static final String WINHELP = "winhelp";

    /** Windows HTMLHelp. Uses &quot;.CHM&quot; files. */ 
    public static final String HTMLHELP = "htmlhelp";

    /** Included for completeness but not currently supported. */
    public static final String HPSHELP = "hpshelp";

    /**
     * Disable help; ignore attempts to set the help filename or invoke
     * help at a specific topic. This may be useful for debugging or
     * running the client without help.
     */
    public static final String NONE = "none";

    private static void error(String message)
    {
        Log.fatalError("transmute.runtime.Help", message);
    }

    private static void information(String message)
    {
        Log.information("transmute.runtime.Help", message);
    }

    private static void warning(String message)
    {
        Log.warning("transmute.runtime.Help", message);
    }

    // This should only be accessed from getDirName()
    private static String dirName;
    private static String getDirName()
    {
        if (dirName == null)
        {
            dirName = Config.instance().getString(HELP_DIR_KEY);
            if (dirName == null)
                error(HELP_DIR_KEY + " property is not set");

            File f = new File(dirName);

            if (!f.exists() || !f.isDirectory())
                error(HELP_DIR_KEY + " property value, " + dirName
                    + ", does not point to a directory");
        }
        return dirName;
    }

    // Should only be accessed from getHelpStyle()
    private static String helpStyle;
    private static String getHelpStyle()
    {
        if (helpStyle == null)
        {
            helpStyle = Config.instance().getString(HELP_SYTLE_KEY);
            if (helpStyle == null)
                error(HELP_SYTLE_KEY + " property is not set");

            if (helpStyle.equals(NONE))
            {
                helpStyle = NONE;
                warning("help system has been disabled");
            }
            else if (helpStyle.equals(WINHELP)) helpStyle = WINHELP;
            else if (helpStyle.equals(HTMLHELP)) helpStyle = HTMLHELP;
            else if (helpStyle.equals(HPSHELP)) helpStyle = HPSHELP;
            else
                error("Illegal " + HELP_SYTLE_KEY + " value: " + helpStyle);
        }
        return helpStyle;
    }

    /**
     * Sets the current help filename. The next call to {@link
     * #showHelpTopic} will use the specified filename.
     */
    public static boolean setHelpFileName(String filename)
    {
        if (getHelpStyle() == NONE)
        {
            warning("ignoring attempt to set help file name");
            return true;
        }

        // perform sanity checking here.
        File file = new File(getDirName(), filename);

        if (!file.exists() || !file.isFile())
            error("cannot open helpfile " + file.getPath());

        if (getHelpStyle() == WINHELP
            && !file.getName().toLowerCase().endsWith(".hlp"))
            error(filename + " is not a winhelp file");
        else if (getHelpStyle() == HTMLHELP
            && !file.getName().toLowerCase().endsWith(".chm"))
            error(filename + " is not a htmlhelp file");
        else if (getHelpStyle() == HPSHELP
            && !file.getName().toLowerCase().endsWith(".hlp"))
            error(filename + " is not a hps help file");

        helpFilename = file.getPath();

        return true;
    }
    private static String helpFilename
        = Config.instance().getString(HELP_DEFAULT_FILE);

    private static boolean libraryLoaded = false;
    private static void loadLibrary()
    {
        if (!libraryLoaded)
        {
            System.loadLibrary("Help");
            libraryLoaded = true;
        }
    }

    /**
     * Launches the help system using the current help filename at the
     * specified topic key.
     */
    public static boolean showHelpTopic(String topic)
    {
        if (getHelpStyle() == NONE)
        {
            warning("ignoring attempt to show a help topic");
            return true;
        }

        if (helpFilename == null)
            error("can't show a help topic before a helpfile has been set");

        if (getHelpStyle() == WINHELP)
        {
            loadLibrary();
            information("showWinHelpTopic() helpFilename = "
                + helpFilename + "  topic = " + topic);
                
            return showWinHelpTopic(helpFilename, topic);
        }
        else if (getHelpStyle() == HPSHELP)
            error("Native Hps help is not currently supported");
        else if (getHelpStyle() == HTMLHELP)
        {
            loadLibrary();
            information("showHtmlHelpTopic() helpFilename = "
                + helpFilename + "  topic = " + topic);
            return showHtmlHelpTopic(helpFilename, topic);
        }
        
        return false;
    }

    private static native boolean showHtmlHelpTopic(
        String filename, String topic);

    private static native boolean showWinHelpTopic(
        String filename, String topic);
}
