package com.ibm.ie.reeng.rt.client;

import com.ibm.ie.reeng.rt.common.Log;
import com.ibm.ie.reeng.rt.common.Sqlca;
import com.ibm.ie.reeng.rt.common.Config;

import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Connection;
import java.sql.DriverManager;


/**
 * This class represents a Database object. It is written using the
 * singleton pattern. The E/SQL style of Hps uses the SQLCA to write control
 * logic for interaction with a database. JDBC (which is CLI) style uses
 * a combination of direct interation via methods and exception
 * handling. This class includes methods to handle this difference by
 * converting <CODE>SQLException</CODE>s to updates to the Sqlca
 * global view.
 */
public class Database
{
    /**
     * Singleton access.
     */
    public static Database instance() { return instance; }

    private static Database instance = new Database();


    /**
     * We only support one connection at a time.
     */
    public static Connection connection()
    {
        return instance.getConnection();
    }

    public static Sqlca getSqlca() { return instance.sqlca; }

    private Database()
    { 
    }

    private Connection con = null;

    private Sqlca sqlca = new Sqlca();

    /**
     * Set up the connection if it hasn't already been set up.
     */
    public Connection getConnection()
    {
        if (con != null) return con;

        String driverClassName = Config.instance().getString("db.driverclass");
        String databaseURL = Config.instance().getString("db.url");
        String userId = Config.instance().getString("db.user");
        String password = Config.instance().getString("db.password");

        String initialSQL = Config.instance().getString("db.startupsql");

        Log.information(this, "JDBC driver class = " + driverClassName);
        Log.information(this, "Database URL = " + databaseURL);
        Log.information(this, "User ID = " + userId);
        Log.information(this, "Password = " + password);

        // db.driverclass is a required property.
        if (driverClassName == null)
            Log.fatalError(this, "No JDBC driver property set.");
        if (databaseURL == null)
            Log.fatalError(this, "No JDBC database URL property set.");

        if (userId == null || password == null)
        {
            // Pop up a dialog window prompting the user to fill in all
            // user name and password.
            LogonDialog details
                = new LogonDialog("Database logon", userId, password);
            
            userId = details.getUserName();
            password = details.getPassword();
        }

        try
        {
            Class.forName(driverClassName).newInstance();
            Log.information(this, "loaded JDBC driver class: "
                + driverClassName);

            con = DriverManager.getConnection(databaseURL, userId, password);
            Log.information(this, "connected as " + userId);

            if (initialSQL != null && !initialSQL.trim().equals(""))
            {
                Log.information(this, "executing startup SQL: "
                    + initialSQL);

                Statement statement = con.createStatement();
                statement.executeUpdate(initialSQL);

                Log.information(this, "startup SQL successfully executed.");
            }
        }
        catch (SQLException e)
        {
            sqlca.updateWith(e);
        }
        catch (Exception e)
        {
            Log.fatalException(e);
        }
        return con;
    }
}
