package com.ibm.ie.reeng.rt.client;

import javax.swing.JComponent;

import java.util.LinkedList;
import java.util.Iterator;

/**
 * Subclasses of this class act as a bridge between GUI elements 
 * and {@link ClientData} instances. See {@link ClientData} for a
 * general discussion.
 * <p>
 * While {@link ClientData} instances are effectively provide a get/set
 * wrapper around a member variable, <code>DataHandler</code> instances
 * mediate between the GUI element and the associated {@link ClientData}
 * instance.
 * <p>
 * <code>DataHandler</code>s also provide identification
 * functionality for the Hps notion of a "data link". This is required
 * by many of the system components (see for example
 * {@link transmute.runtime.system.component.SetFieldMode}).
 * <p>
 * The property of causing immediate return if the data is updated and
 * the property of being mandatory for the datalink are represented in
 * this class.
 * <p>
 * Note that for system components that return a VIEW_LONG_NAME and
 * FIELD_LONG_NAME in their output view:
 * <UL>
 * <LI> The FIELD_LONG_NAME is the terminating field of the element's data
 *   link and the VIEW_LONG_NAME is its immediate containing view unless
 *   the field is contained by a multiply occuring view (which need not
 *   be its immediate parent) in which case this view is used.
 * <LI> In the case of combo boxes which may have two datalinks, one for
 *   the currently selected item and one for the contents of the drop
 *   down list, the data link for the drop down list, a.k.a. the domain,
 *   is used.
 * <LI> Remember that the returned VIEW_LONG_NAME and FIELD_LONG_NAME may
 *   not necessarily uniquely identify the user interface element within
 *   the window.
 * </UL>
 * For system components that take a VIEW_LONG_NAME and FIELD_LONG_NAME
 * in their input view:
 * <UL>
 * <LI> The VIEW_LONG_NAME or FIELD_LONG_NAME (but not both) can be left
 *   blank and will match against any view or field respectively.
 * 
 * <LI> When specifying the VIEW_LONG_NAME any view that is a descendant of
 *   the window can be used, it doesn't have to be the immediate
 *   containing view of a view, e.g. it can be set to the immediate view
 *   child of the window and FIELD_LONG_NAME can be left blank and this
 *   combination will match every user interface element in the window
 *   that has a data link.
 * 
 * <LI> In the case of combo boxes VIEW_LONG_NAME and FIELD_LONG_NAME can
 *   match against either of its data links.
 * </UL>
 */
public abstract class DataHandler
{
    public final static int INVALID = 0;
    public final static int VALID = 1;
    public final static int EMPTY = 2;


    /**
     * Classes (i.e. GUI element classes) which are associated with
     * <code>DataHandler</code> instances implement this interface.
     */
    public static interface Interface
    {
        public DataHandler getDataHandler();

        public void setClientData(ClientData clientData);
    }


    private boolean immediateReturn = false;
    private boolean mandatory = false;

    protected JComponent component;
    protected ClientData clientData;

public DataHandler()
{
}

    public DataHandler(JComponent component)
    {
        this.component = component;
    }


    public void setMandatory() { mandatory = true; }

    public void setMandatory(boolean mandatory)
    {
        this.mandatory=mandatory;
    }


    public boolean getMandatory() { return mandatory; }


    public void setImmediateReturn() { immediateReturn = true; }
    
    public void setImmediateReturn(boolean immediateReturn) 
    { 
        this.immediateReturn = immediateReturn; 
    }

    public boolean getImmediateReturn()
    {
        return immediateReturn;
    }


    public void setClientData(ClientData clientData)
    {
        this.clientData = clientData;
    }

    public ClientData getClientData()
    {
        return(this.clientData);
    }
    
    protected void stateChangedHandler(int storeResult) { }

    
    protected void stateChanged()
    {
        int result = storeData();

        stateChangedHandler(result);

        //specifies new AlteredField to TransmuteDialog...
        dialog.setAlteredField(new AlteredField(getPrimaryView(), getPrimaryField()));

        if (result != INVALID)
        {
            if (clientData != null)
                getDialog().updateSharedDatalinks(this);

            if (immediateReturn)
            {
                String id = ((Identifier)component).getIdentifier();
                getDialog().dispatchImmediateReturnEvent(id);
            }
        }
    }


    // ------------------------------------------------------------------------


    private TransmuteDialog dialog;


    protected TransmuteDialog getDialog()
    {
        if (dialog == null)
            dialog = TransmuteDialog.getParentDialog(component);

        return dialog;
    }


    // ------------------------------------------------------------------------


    public abstract void displayData();


    public abstract int storeData();


    // ------------------------------------------------------------------------


    // See notes on long names above.


    private String primaryName;
    private String primaryView;
    private String primaryField;



    private LinkedList longNameList = new LinkedList();


    public void addLongName(String name)
    {
        LongName longName = new LongName(name);

        if (primaryName == null)
        {
            primaryName = name;
            primaryView = longName.getFirstView();
            primaryField = longName.getField(); 
        }

        longNameList.add(longName);
    }


    public boolean matchLongName(String view)
    {
        return matchLongName(view, "");
    }


    public boolean matchLongName(String view, String field)
    {
        if (view.equals("") && field.equals("")) return false;

        Iterator iterator = longNameList.iterator();

        while (iterator.hasNext())
            if (((LongName)iterator.next()).matches(view, field))
                return true;

        return false;
    }

    public String getPrimaryName() { return primaryName; }

    public String getPrimaryView() { return primaryView; }

    public String getPrimaryField() { return primaryField; }
}
