package com.ibm.ie.reeng.rt.common;

import com.ibm.ie.reeng.rt.common.Log;
import com.ibm.ie.reeng.rt.common.Config;

import java.text.SimpleDateFormat;
import java.text.ParsePosition;
import java.util.Calendar;

import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;


/**
 * Since we are using ISO dates, times and timestamps (java.sql.Date,
 * java.sql.Time and java.sql.Timestamp respectively) we have to be very
 * careful using UTC time (via getTime()) on such instances to do
 * calculations. Implicitly the ISO temporal types are relative to a
 * Calendar (and timezone). Because of this, for example, we cannot
 * assume the length of a day is 1000*60*60*24 milliseconds because
 * adjustments like daylight savings provide exceptions. Therefore we
 * must verify that any calculation or manipulation of the millisecond
 * portion of such types are immune from calendar effects.
 * <p>
 * Note that the Java epoch began on Jan 1, 1970 whereas the HPS epoch
 * began on Jan 1, 0000.  Note that this date is completely fictional
 * and doesn't have an ISO counterpart.  Also note that Hps actually
 * uses a variant of the Gregorian proleptic calendar which is never
 * used (unlike the Julian proleptic calendar which is universally used
 * for BCE era dates). Therefore the correspondance between ISO and Hps
 * dates breaks down prior to the introduction of the Gregorian calandar
 * in 1582. We do not attempt to perform the complex calculations for
 * dates prior to 1582. Instead we map Hps 0 to ISO 0001-01-01 (the
 * first day of the current era). 
 * <p>
 * Basically we ensure that:
 * <code>oldToNewDate(newToOldDate(isoDate)).equals(isoDate)</code> for
 * all ISO dates. However we can only ensure that 
 * <code>newToOldDate(oldToNewDate(hpsDate)) == hpsDate</code> for
 * hpsDates outside of the range HPS_JULIAN_END..HPS_GREGORIAN_START
 * (see source for these values) and greater than or equal to -1. Note
 * that this is optimal in that it minimises the range of values for
 * which the mapping to ISO dates break down while preserving a mapping
 * for Hps dates around 0 (which is important because Hps rule logic
 * regularly uses such dates in comparisons and/or to indicate an
 * invalid date).
 */
public class TemporalFunctions
{
    private final static Calendar DEFAULT_CALENDAR;

    public final static Date ZERO_DATE;
    public final static Time ZERO_TIME;
    public final static Timestamp ZERO_TIMESTAMP;

    public final static Date INVALID_DATE;
    public final static Time INVALID_TIME;

    private final static Date GREGORIAN_START;
    private final static int HPS_GREGORIAN_START = 578102;
    private final static int HPS_JULIAN_END = 577724;


    /**
     * Easier to handle ordering dependencies by explicitly calculating
     * the final static constants.
     */
    static
    {
        // Use the default locale calendar.
        DEFAULT_CALENDAR = Calendar.getInstance();

        // start of the current era.
        ZERO_DATE = Date.valueOf("0001-01-01");
        
        // "zero" o'clock.
        ZERO_TIME = Time.valueOf("00:00:00");

        // start of the current era.
        ZERO_TIMESTAMP = new Timestamp(ZERO_DATE.getTime());

        // these reflect the discontinuity between Hps and Iso dates.
        GREGORIAN_START = Date.valueOf("1582-10-15");

        INVALID_DATE = oldToNewDate(-1);
        INVALID_TIME = oldToNewTime(-1);
    }


    public static boolean isInvalidDate(Date date)
    {
        return date.getTime() < ZERO_DATE.getTime();
    }


    public static boolean isInvalidTime(Time time)
    {
        return time.getTime() < ZERO_TIME.getTime();
    }


    // ------------------------------------------------------------------------

    /**
     * Returns the number of days from an arbitrary epoch to 1 January
     * of the given year using the proleptic Gregorian calendar. We
     * don't care what the epoch is because we only use it for offset
     * purposes.
     * <p>
     * It should always return positive.
     */
    private static int days(int year)
    {
        year = year - 1;
        return 365 + year * 365 + (year / 4) - (year / 100) + (year / 400);
    }

    /**
     * After the Gregorian switch over, Hps and ISO linearly correspond.
     */
    public static int newToOldDate(Date date)
    {
        if (date.equals(INVALID_DATE)) return -1;

        Calendar cal = getCalendar(date);
        int year = cal.get(Calendar.YEAR);
        int dayOfYear = cal.get(Calendar.DAY_OF_YEAR);

        //1.2KL if (date.compareTo(GREGORIAN_START) >= 0)
        if (!date.before(GREGORIAN_START))
        {
            cal = getCalendar(GREGORIAN_START);
            int gregYear = cal.get(Calendar.YEAR);
            int gregDayOfYear = cal.get(Calendar.DAY_OF_YEAR);

            return days(year) + dayOfYear - days(gregYear) - gregDayOfYear
                + HPS_GREGORIAN_START;
        }

        // At this point the correspondance breaks down. We preserve the
        // mapping back and forth but not the date semantics. (For
        // example any Hps date before 367 cannot be mapped to ISO.)
        // 
        // Hps date 0 maps to 0001-01-01.

        cal = getCalendar(ZERO_DATE);
        int firstYear = cal.get(Calendar.YEAR);
        int firstDayOfYear = cal.get(Calendar.DAY_OF_YEAR);

        return days(year) + dayOfYear - days(firstYear) - firstDayOfYear;
    }

    public static Date oldToNewDate(int days)
    {
        Calendar cal;

        if (days >= HPS_GREGORIAN_START)
        {
            cal = getCalendar(GREGORIAN_START);
            days = days - HPS_GREGORIAN_START;
        }
        else if (days <= HPS_JULIAN_END)
        {
            cal = getCalendar(ZERO_DATE);
        }
        else
        {
            Log.error("TemporalFunctions",
                "cannot map Hps date to ISO date");
            return null;
        }

        int year = cal.get(Calendar.YEAR);
        int dayOfYear = cal.get(Calendar.DAY_OF_YEAR);

        int fictDays = days(year) + days + dayOfYear;

        int resultYear = (int)(fictDays / 365.2425);
        
        int resultDayOfYear = fictDays - days(resultYear);

        cal.set(Calendar.YEAR, resultYear);
        cal.set(Calendar.DAY_OF_YEAR, resultDayOfYear);
        
        return new Date(cal.getTime().getTime());
    }


    public static int newToOldTime(Time time)
    {
        if (isInvalidTime(time)) return -1;

        Calendar cal = getCalendar(time);
        int hours = cal.get(Calendar.HOUR_OF_DAY);
        int minutes = cal.get(Calendar.MINUTE);
        int seconds = cal.get(Calendar.SECOND);
        int millis = cal.get(Calendar.MILLISECOND);

        return millis + 1000 * (seconds + 60 * (minutes + 60 * hours));
    }

    public static Time oldToNewTime(int hpsMillis)
    {
        Calendar cal = getCalendar();
        cal.set(Calendar.YEAR, 1970);
        cal.set(Calendar.MONTH, Calendar.JANUARY);
        cal.set(Calendar.DAY_OF_MONTH, 1);

        if (hpsMillis > 0)
        {
            int millis = hpsMillis % 1000;
            int hpsSeconds = hpsMillis / 1000;
            int seconds = hpsSeconds % 60;
            int hpsMinutes = hpsSeconds / 60;
            int minutes = hpsMinutes % 60;
            int hours = hpsMinutes / 60;

            cal.set(Calendar.HOUR_OF_DAY, hours);
            cal.set(Calendar.MINUTE, minutes);
            cal.set(Calendar.SECOND, seconds);
            cal.set(Calendar.MILLISECOND, millis);
        }
        else
        {
            // arbitrary encoding for invalid time.
            return new Time(cal.getTime().getTime() - hpsMillis);
        }

        return new Time(cal.getTime().getTime());
    }


    // ------------------------------------------------------------------------


    // The various flavours of the Hps DATE function:


    public static Date date()
    {
        return new Date(System.currentTimeMillis());
    }


    public static Date date(int days)
    {
        return oldToNewDate(days);
    }


    public static Date date(Timestamp timestamp)
    {
        return new Date(timestamp.getTime());
    }


    public static Date date(String rep)
    {
        return date(rep, getDefaultDateFormat());
    }


    /**
     * This function tries to parse a date string according to a given
     * format, like the HPS original if this proves unsuccessful it
     * tries to parse the string again with all separator characters
     * removed from the format string. If the year is represented as two
     * digits in the string the century is infered - see correctCentury().
     * <p>
     * Note: at the moment if the user enters a four digit year before
     * 100 AD, e.g. 0010, this is currently misinterpreted as a two digit
     * year, e.g. 0010 becomes 10 and addCentury() is called on it.
     */
    public static Date date(String rep, String fmt)
    {
        Date date = stringToSqlDate(rep, fmt);

        if (date != null) return date;

        // Try again with no separators.
        date = stringToSqlDate(rep, stripSeparators(fmt));

        if (date != null) return date;

        return INVALID_DATE;
    }


    // ------------------------------------------------------------------------


    // The various flavours of the Hps TIME function:


    public static Time time()
    {
        Calendar cal = getCalendar(System.currentTimeMillis());

        cal.set(Calendar.YEAR, 1970);
        cal.set(Calendar.MONTH, Calendar.JANUARY);
        cal.set(Calendar.DAY_OF_MONTH, 1);

        return new Time(cal.getTime().getTime());
    }


    public static Time time(int millis)
    {
        return oldToNewTime(millis);
    }


    public static Time time(Timestamp timestamp)
    {
        // !!! DANGER UTC time:
        // Note: milli =  10 ^ -3, nano  =  10 ^ -9
        // To convert from nanos to mills divide by 10 ^ 6
        long utc = timestamp.getTime() + (timestamp.getNanos() / 1000000);
        Calendar cal = getCalendar(utc);

        cal.set(Calendar.YEAR, 1970);
        cal.set(Calendar.MONTH, Calendar.JANUARY);
        cal.set(Calendar.DAY_OF_MONTH, 1);

        return new Time(cal.getTime().getTime());
    }

    public static Time time(String rep)
    {
        return time(rep, getDefaultTimeFormat());
    }


    /**
     * This function tries to parse a time string according to a given
     * format. It works like <code>date(...)</code> and tries to parse
     * the time without separators if the initial parse fails.
     */
    public static Time time(String rep, String fmt)
    {
        Time time = stringToSqlTime(rep, fmt);

        if (time != null) return time;

        // Try again with no separators.
        time = stringToSqlTime(rep, stripSeparators(fmt));

        if (time != null) return time;

        return INVALID_TIME;
    }


    // ------------------------------------------------------------------------


    // Date/Time parsing/formatting functions.

    private final static String FALLBACK_DATE_FORMAT = "%0d/%0m/%Y";
    private final static String FALLBACK_TIME_FORMAT = "%0t:%0m:%0s";

    private static String defaultDateFormat;
    private static String defaultTimeFormat;


    private static String getDefaultDateFormat()
    {
        if (defaultDateFormat == null)
        {
            defaultDateFormat = Config.instance().getString("format.date");
            if (defaultDateFormat == null)
                defaultDateFormat = FALLBACK_DATE_FORMAT;
        }

        return defaultDateFormat;
    }


    private static String getDefaultTimeFormat()
    {
        if (defaultTimeFormat == null)
        {
            defaultTimeFormat = Config.instance().getString("format.time");
            if (defaultTimeFormat == null)
                defaultTimeFormat = FALLBACK_TIME_FORMAT;
        }

        return defaultTimeFormat;
    }


    private static String stripSeparators(String format)
    {
        final String SEPARATORS = " /,-:";

        StringBuffer result = new StringBuffer();
        char[] original = format.toCharArray();

        for (int i = 0; i < original.length; i++)
        {
            if (SEPARATORS.indexOf(original[i]) == -1)
                result.append(original[i]);
        }

        return result.toString();
    }


    /**
     * Parse the passed in string according to the given Hps format.
     */
    private static Date stringToSqlDate(String dateString,
        String hpsFormat)
    {
        String dateFormat = convertDateFormat(hpsFormat);

        SimpleDateFormat formatter = new SimpleDateFormat(dateFormat);

        java.util.Date date = strictParse(formatter, dateString);

        if (date == null) return null;

        if (requiresCenturyCorrection(hpsFormat))
            return correctCentury(new Date(date.getTime()));
        else return new Date(date.getTime());
    }

    /**
     * Determines whether century correction is required on a parsed
     * date. As far as I know, it is only required for 2 digit year
     * formats.
     */
    private static boolean requiresCenturyCorrection(String hpsFormat)
    {
        return hpsFormat.indexOf("%y") >= 0 || hpsFormat.indexOf("%0y") >= 0;
    }

    /**
     * Parse the passed in string according to the given Hps format.
     */
    private static Time stringToSqlTime(String timeString,
        String hpsFormat)
    {
        String timeFormat = convertTimeFormat(hpsFormat);

        SimpleDateFormat formatter =
            new SimpleDateFormat("yyyy/MM/dd " + timeFormat);

        java.util.Date date =
            strictParse(formatter, "0001/01/01 " + timeString);


        if (date == null) return null;

        Calendar cal = getCalendar(date);

        cal.set(Calendar.YEAR, 1970);
        cal.set(Calendar.MONTH, Calendar.JANUARY);
        cal.set(Calendar.DAY_OF_MONTH, 1);

        return new Time(cal.getTime().getTime());
    }


    // ------------------------------------------------------------------------


    private static Date correctCentury(Date date)
    {
        // HPS takes a 100 year sliding window approach to the Y2K problem.
        //
        // Any two digit date will be mapped into the 100 years period
        // ending 30 years after the current year.
        //
        // e.g. if the current year is 1999 then any two digit date will be
        // mapped into the period 1930 to 2029, so:
        //  30 becomes 1930
        //  55 becomes 1955
        //  15 becomes 2015
        //  29 becomes 2029.
        //

        Calendar calendar = getCalendar(System.currentTimeMillis());

        int currentYear = calendar.get(Calendar.YEAR);

        calendar.setTime(date);

        int year = calendar.get(Calendar.YEAR);

        if (year > 99) return date;

        int currentCentury = currentYear - (currentYear % 100);

        int start = currentYear - 69;

        int end = start + 99;

        year += currentCentury;

        if (year < start) year += 100;
        else if (year > end) year -= 100;

        calendar.set(Calendar.YEAR, year);

        return new Date(calendar.getTime().getTime());
    }


    private static java.util.Date strictParse(SimpleDateFormat formatter,
        String string)
    {
        if (string == null || string.length() == 0) return null;

        ParsePosition pos = new ParsePosition(0);

        formatter.setLenient(false);

        java.util.Date date = formatter.parse(string, pos);

        //1.2KL if (pos.getIndex() == string.length() && pos.getErrorIndex() == -1)
        if (pos.getIndex() == string.length())
            return date;

        return null;
    }


    // ------------------------------------------------------------------------


    private static String convertDateFormat(String hpsFormat)
    {
        return convertFormatString(hpsFormat, true);
    }


    private static String convertTimeFormat(String hpsFormat)
    {
        return convertFormatString(hpsFormat, false);
    }


    private static String convertFormatString(String hpsFormat, boolean date)
    {
        char[] original = hpsFormat.toCharArray();
        StringBuffer result = new StringBuffer();
        int i = 0;

        while (i < original.length)
        {
            char c = original[i++];

            if (c == '%')
            {
                String token = String.valueOf(original[i++]);

                if (token.charAt(0) == '0') token += original[i++];

                if (date) result.append(convertDateFormatToken(token));
                else result.append(convertTimeFormatToken(token));
            }
            else result.append(c);
        }

        return result.toString();
    }
    

    private static String convertTimeFormatToken(String token)
    {
        if (token.equals("h")) return "h";
        else if (token.equals("0h")) return "hh";
        else if (token.equals("t")) return "H";
        else if (token.equals("0t")) return "HH";
        else if (token.equals("m")) return "m";
        else if (token.equals("0m")) return "mm";
        else if (token.equals("s")) return "s";
        else if (token.equals("0s")) return "ss";
        else if (token.equals("f")) return "S";
        else if (token.equals("0f")) return "SSS";
        else if (token.equals("x")) return "a";
        else
        {
            // Time format tokens with no direct Java equivalent.

            if (token.equals("H")) // H is hour as word.
            {
                Log.warning("TemporalFunctions", "treating 'H' as 'h'");
                return "h";
            }
            if (token.equals("M")) // M is minute as word.
            {
                Log.warning("TemporalFunctions", "treating 'M' as 'm'");
                return "m";
            }
            if (token.equals("S")) // S is second as word.
            {
                Log.warning("TemporalFunctions", "treating 'S' as 's'");
                return "s";
            }
            else
            {
                Log.fatalError("TemporalFunctions",
                    "unknown time format token '" + token + "'");
                return null;
            }
        }
    }


    private static String convertDateFormatToken(String token)
    {
        if (token.equals("m")) return "M";
        else if (token.equals("0m")) return "MM";
        else if (token.equals("M")) return "MMMM";
        else if (token.equals("d")) return "d";
        else if (token.equals("0d")) return "dd";
        else if (token.equals("j")) return "D";
        else if (token.equals("0j")) return "DD";
        else if (token.equals("0y")) return "yy";
        else if (token.equals("Y")) return "yyyy";
        else if (token.equals("W")) return "EEEE";
        else
        {
            // Date format tokens with no direct Java equivalent.

            // D is ordinal day of month, e.g. 1st.
            if (token.equals("D"))
            {
                Log.warning("TemporalFunctions", "treating 'D' as 'd'");
                return "d";
            }
            else if (token.equals("c")) // c is century.
            {
                Log.warning("TemporalFunctions", "treating 'c' as 'Y'");
                return "yyyy";
            }
            else if (token.equals("0c")) // c is zero padded century.
            {
                Log.warning("TemporalFunctions", "treating '0c' as 'Y'");
                return "yyyy";
            }
            else if (token.equals("y")) // y is non-padded year of century.
            {
                Log.warning("TemporalFunctions", "treating 'y' as '0y'");
                return "yy";
            }
            else
            {
                Log.fatalError("TemporalFunctions",
                    "unknown date format token '" + token + "'");
                return null;
            }
        }
    }


    // ------------------------------------------------------------------------


    // Functions for explicit interval arithmetic


    /**
     * Note these reflect wonky Hps semantics.
     */
    public static int daysBetween(Date end, Date start)
    {
        return newToOldDate(end) - newToOldDate(start);
    }


    public static int millisBetween(Time end, Time start)
    {
        return newToOldTime(end) - newToOldTime(start);
    }


    public static Date addDaysToDate(int days, Date date)
    {
        return oldToNewDate(newToOldDate(date) + days);
    }


    public static Time addMillisToTime(int millis, Time time)
    {
        return oldToNewTime(newToOldTime(time) + millis);
    }


    // ------------------------------------------------------------------------


    // The various flavours of the Hps CHAR function:


    public static String toString(Date date)
    {
        return toString(date, getDefaultDateFormat());
    }


    public static String toString(Date date, String hpsFormat)
    {
        if (isInvalidDate(date)) return "-1";

        SimpleDateFormat formatter =
            new SimpleDateFormat(convertDateFormat(hpsFormat));

        return formatter.format(date);
    }


    public static String toString(Time time)
    {
        return toString(time, getDefaultTimeFormat());
    }


    public static String toString(Time time, String hpsFormat)
    {
        if (isInvalidTime(time)) return "-1";

        SimpleDateFormat formatter =
            new SimpleDateFormat(convertTimeFormat(hpsFormat));

        return formatter.format(time);
    }


    // ------------------------------------------------------------------------


    private static Calendar getCalendar(java.util.Date date)
    {
        DEFAULT_CALENDAR.clear();
        DEFAULT_CALENDAR.setTime(date);

        return DEFAULT_CALENDAR;
    }


    private static Calendar getCalendar(long millis)
    {
        return getCalendar(new java.util.Date(millis));
    }

    private static Calendar getCalendar()
    {
        DEFAULT_CALENDAR.clear();

        return DEFAULT_CALENDAR;
    }

    // ------------------------------------------------------------------------


    // Date component extraction functions:


    public static int dayOfMonth(Date date)
    {
        return getCalendar(date).get(Calendar.DAY_OF_MONTH);
    }


    public static int dayOfYear(Date date)
    {
        return getCalendar(date).get(Calendar.DAY_OF_YEAR);
    }


    public static int dayOfWeek(Date date)
    {
        return getCalendar(date).get(Calendar.DAY_OF_WEEK);
    }


    public static int month(Date date)
    {
        return getCalendar(date).get(Calendar.MONTH) + 1;
    }


    public static int year(Date date)
    {
        return getCalendar(date).get(Calendar.YEAR);
    }


    // ------------------------------------------------------------------------


    // Time component extraction functions:


    public static int hours(Time time)
    {
        return getCalendar(time).get(Calendar.HOUR_OF_DAY);
    }


    public static int minutes(Time time)
    {
        return getCalendar(time).get(Calendar.MINUTE);
    }


    public static int seconds(Time time)
    {
        return getCalendar(time).get(Calendar.SECOND);
    }


    public static int milSecs(Time time)
    {
        return getCalendar(time).get(Calendar.MILLISECOND);
    }


    public static int minutesOfDay(Time time)
    {
        return hours(time) * 60 + minutes(time);
    }


    public static int secondsOfDay(Time time)
    {
        return minutesOfDay(time) * 60 + seconds(time);
    }


    // ------------------------------------------------------------------------


    // The two timestamp functions.


    /**
     * Note java.sql.Timestamps only provide nanosecond accuracy. Hps
     * doesn't seem to guarantee anything regarding what the fraction
     * part represents but suggests that it should generally be
     * picoseconds. This is what is assumed here. As a result some
     * accuracy is actually lost. Thus you cannot assume that 
     * <code>timestamp(d, t, f).fraction()</code> will equal
     * <code>f</code>.
     * <p>
     * Note that ISO dates don't fully support Hps semantics. For
     * example, if you create a Hps timestamp with an invalid time
     * component, you will not get the invalid time back using the
     * time(Timestamp ts) method. Hps stores timestamps as three
     * completely independent components while ISO timestamps consist of
     * a single compounded date, time and nano-second.
     */
    public static Timestamp timestamp(Date date, Time time, int fraction)
    {
        int year = getCalendar(date).get(Calendar.YEAR);
        int dayOfYear = getCalendar(date).get(Calendar.DAY_OF_YEAR);
        int hours = getCalendar(time).get(Calendar.HOUR_OF_DAY);
        int minutes = getCalendar(time).get(Calendar.MINUTE);
        int seconds = getCalendar(time).get(Calendar.SECOND);
        int millis = getCalendar(time).get(Calendar.MILLISECOND);

        // First construct the date/time portion.
        Calendar cal = getCalendar();
        cal.set(Calendar.YEAR, year);
        cal.set(Calendar.DAY_OF_YEAR, dayOfYear);
        cal.set(Calendar.HOUR_OF_DAY, hours);
        cal.set(Calendar.MINUTE, minutes);
        cal.set(Calendar.SECOND, seconds);
        cal.set(Calendar.MILLISECOND, millis);

        // the timestamp constructor truncates the millis and stores
        // them in the nanos component.
        Timestamp timestamp = new Timestamp(cal.getTime().getTime());

        // fractionNanos holds the fraction argument as nanoseconds.
        int fractionNanos = fraction / 1000;

        timestamp.setNanos(fractionNanos + timestamp.getNanos());

        return timestamp;
    }


    public static Timestamp timestamp()
    {
        return new Timestamp(System.currentTimeMillis());
    }


    /**
     * I keep forgetting this so I'm writing this here:
     *      milli   10 ^ -3
     *      nano    10 ^ -9
     *      pico    10 ^ -12
     * We have to be careful below to avoid overflow. 
     * <p>
     * Note <code>java.sql.Timestamp</code>s allow access to the nano
     * seconds after a particular second. This maximum is 999,999,999.
     * Stripping the milliseconds gives 999,999.
     */
    public static int fraction(Timestamp timestamp)
    {
        int nanos = timestamp.getNanos();
        int fract = nanos % 1000000;    // strip any milliseconds
        return fract * 1000;            // return as picoseconds
    }
}
