package com.ibm.ie.reeng.rt.common;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;

import java.util.HashMap;
import java.util.ArrayList;


/**
 * This is a base class for all classes corresponding to Hps sets. Such
 * Hps entities map to Java classes with <code>public static
 * final</code> members corresponding to the children of the original
 * Hps sets. However Hps sets allow "dynamic" lookup (i.e. by name
 * represented as a string) and reverse lookup (i.e. by value)
 * using setencoding and setdisplay.  Rather than using reflection each
 * time one of these functions are used, the constructor for this
 * classe builds forward and reverse maps between names and values using
 * reflection.
 * <p>
 * In addition sets are used in windows for combo-boxes for example;
 * this base class provides support methods for this usage of sets. In
 * this case the values of a set are associated with a "display value"
 * or a human readable version of the value.  However the implementation
 * of these methods here is relatively trivial; values represent
 * themselves. {@link LookupSet}s allow the definition of
 * distinct "display strings" for the values.
 */
public class Set
{
    private HashMap nameToValue;
    private HashMap valueToName;
    protected ArrayList nameList;
    private ArrayList valueList;


    /**
     * This protected constructor uses reflection to find the name/value
     * pairs in the set and create maps from one to the other.
     */
    protected Set()
    {
        try
        {
            nameList = new ArrayList();
            valueList = new ArrayList();

            nameToValue = new HashMap();
            valueToName = new HashMap();

            Field[] fields = getClass().getFields();
            final int MODIFIERS =
                Modifier.PUBLIC | Modifier.STATIC | Modifier.FINAL;

            for (int i = 0; i < fields.length; i++)
            {
                String name = fields[i].getName();

                int modifiers = fields[i].getModifiers();

                if (modifiers == MODIFIERS)
                {
                    Object value = fields[i].get(null);

                    nameToValue.put(name, value);
                    valueToName.put(value, name);

                    //1.2KL nameList.add(name);
                    nameList.add(name);
                    //1.2KL valueList.add(value);
                    valueList.add(value);
                }
            }
        }
        catch (Exception e)
        {
            Log.fatalException(e);
        }
    }

    /**
     * Gets the name corresponding to a value. Note that doubles and
     * ints have to be wrapped in Double or Integer objects
     * respectively.
     */
    public String getName(Object value)
    {
        if (!valueToName.containsKey(value)) return "";
        else return valueToName.get(value).toString();
    }

    /**
     * Convenience method for getting the name corresponding to a int
     * value.
     */
    public String getName(int value)
    {
        return getName(new Integer(value));
    }

    /**
     * Supports the setencoding function. Note that doubles and ints are
     * wrapped in Double or Integer objects respectively.
     */
    public Object getValue(String name)
    {
        return nameToValue.get(name);
    }

    /**
     * Returns whether a value is contained in this set.
     */
    public boolean inset(Object value)
    {
        return valueToName.containsKey(value);
    }

    /**
     * Convenience method for determining membership of this set.
     */
    public boolean inset(int value)
    {
        return inset(new Integer(value));
    }


    // ------------------------------------------------------------------------

    
    /**
     * Returns the number of values in this set.
     */
    public int size() { return valueList.size(); }

    /**
     * Returns the <code>i</code><sup>th</sup> display string for this
     * set. Note that the default (as defined here) is to simply return
     * the <code>i</code><sup>th</sup> value. {@link LookupSet}s provide
     * more flexibility and allow arbitrary strings to be used to
     * represent user readable versions of the values.
     */
    //1.2KL public Object getDisplay(int i) { return valueList.get(i); }
     public Object getDisplay(int i) { return valueList.get(i); }


    /**
     * Returns the position of a display value in this set. Returns -1
     * if the value is not in this set.
     */
    protected int getPosition(Object display)
    {
        return valueList.indexOf(display);
    }

    /**
     * Returns the "display value" associated with a value in this set.
     */
    public Object valueToDisplay(Object value)
    {
        int i = valueList.indexOf(value);

        if (i != -1) return getDisplay(i);

        // If NOT set member...

        Object sample = getDisplay(0);

        if (value.getClass().isInstance(sample)) return value;
        else return Util.getClearValue(sample);
    }

    /**
     * Returns the values associated with a "display value".
     */
    public Object displayToValue(Object display)
    {
        int i = getPosition(display);

        //1.2KL if (i != -1) return valueList.get(i);
        if (i != -1) return valueList.get(i);

        // If NOT set member...

        //1.2KL Object sample = valueList.get(0);
        Object sample = valueList.get(0);

        if (display.getClass().isInstance(sample)) return display;
        else return Util.getClearValue(sample);
    }
}
