package com.ibm.ie.reeng.rt.common;


import java.util.Arrays;

/**
 * This {@link Marshall} implementation uses a <code>byte</code> buffer.
 * Views and fields are laid out as they are in memory for real Hps
 * clients.
 * <p>
 * Surprisingly this layout does not match that of the C structs
 * generated for C user components; no fields are padded.
 * <p>
 * Some methods have be optimised which comprimises readability.
 */
public class OverlayMarshall extends ByteMarshall
{
    public OverlayMarshall() { }


    public OverlayMarshall(String buffer) { setReadBuffer(buffer); }


    // ------------------------------------------------------------------------


    /**
     * Originally implemented as:
     * <pre>
     *     public Marshall writeChar(String field, int length)
     *     {
     *         put(Util.padRight(field, length, Util.SPACE).getBytes());
     *         return this;
     *     }
     * </pre>
     * Optimised here.
     */
    public Marshall writeChar(String field, int len)
    {
        if (state != WRITE)
            throw new RuntimeException("operation illegal in current state");

        byte[] element = Util.padRight(field, len, Util.SPACE).getBytes();

        int required = length + element.length;

        if (buffer.length < required) resize(required * 2);

        System.arraycopy(element, 0, buffer, length, element.length);

        length += element.length;

        return this;
    }


    public Marshall writeVarchar(String field, int length)
    {
        putSmallint(field.length(), 0, length);

        put(Util.padRight(field, length, Util.SPACE).getBytes());

        return this;
    }


    public Marshall writeSmallint(int field)
    {
        putSmallint(field, Util.MIN_SMALLINT, Util.MAX_SMALLINT);
        return this;
    }


    public Marshall writeInteger(int field)
    {
        putInteger(field, Util.MIN_INTEGER, Util.MAX_INTEGER);
        return this;
    }


    public Marshall writePic(double field, boolean signed,
        int length, int fraction)
    {
        if (signed) length--;
        else if (field < 0) field = 0;

        StringBuffer buffer = new StringBuffer();

        boolean negative = encodeReal(field, length, fraction, buffer);

        if (signed) put((byte)(negative ? '-' : '+'));

        String number = Util.padLeft(buffer.toString(), length, Util.ZERO);

        put(number.getBytes());

        return this;
    }


    // Note: the field length of DEC data items does not include the sign.
    public Marshall writeDec(double field, int length, int fraction)
    {
        StringBuffer buffer = new StringBuffer();

        boolean negative = encodeReal(field, length, fraction, buffer);

        if (negative) buffer.insert(0, '-');

        String number = Util.padLeft(buffer.toString(),
            (length + 1), Util.SPACE);

        put(number.getBytes());

        return this;
    }


    public Marshall writeDate(java.sql.Date field)
    {
        int intDate = TemporalFunctions.newToOldDate(field);
        putInteger(intDate, Util.MIN_INTEGER, Util.MAX_INTEGER);

        return this;
    }


    public Marshall writeTime(java.sql.Time field)
    {
        int intTime = TemporalFunctions.newToOldTime(field);
        putInteger(intTime, Util.MIN_INTEGER, Util.MAX_INTEGER);

        return this;
    }


    public Marshall writeTimestamp(java.sql.Timestamp field)
    {
        int date = TemporalFunctions.newToOldDate(
            TemporalFunctions.date(field));
        int time = TemporalFunctions.newToOldTime(
            TemporalFunctions.time(field));
        int fract = TemporalFunctions.fraction(field);

        putInteger(date, 0, Util.MAX_INTEGER);
        putInteger(time, 0, Util.MAX_TIME);
        putInteger(fract, Util.MIN_INTEGER, Util.MAX_INTEGER);

        return this;
    }


    // ------------------------------------------------------------------------


    private void putInteger(int field, int min, int max)
    {
        if (field < min) field = min;
        else if (field > max) field = max;

        for (int i = 0; i < 4; i++)
        {
            put((byte)field);
            field >>>= 8;
        }
    }


    private void putSmallint(int field, int min, int max)
    {
        if (field < min) field = min;
        else if (field > max) field = max;

        put((byte)field);
        field >>>= 8;
        put((byte)field);
    }


    private boolean encodeReal(double field, int length, int fraction,
        StringBuffer number)
    {
        String string = Util.toDoubleString(field);
        int point = string.indexOf('.');
        String integer = string.substring(0, point);
        String decimal = string.substring((point + 1), string.length());
        boolean negative = integer.startsWith("-");

        if (negative) integer = integer.substring(1, integer.length());

        int digits = length - fraction;

        if (integer.length() > digits)
        {
            char[] nines = new char[digits];

            Arrays.fill(nines, '9');

            integer = new String(nines);
        }

        if (decimal.length() > fraction)
            decimal = decimal.substring(0, fraction);
        else decimal = Util.padRight(decimal, fraction, Util.ZERO);

        number.append(integer);
        number.append(decimal);

        return negative;
    }


    // ------------------------------------------------------------------------


    /**
     * Originally implemented as:
     * <pre>
     *     public String readChar(int length)
     *     {
     *         String field = new String(get(length));
     * 
     *         return Util.unpadRight(field, Util.SPACE);
     *     }
     * </pre>
     * Optimised here.
     */
    public String readChar(int length)
    {
        if (state == WRITE) setReadBuffer(getWriteByteBuffer());

        int remaining = buffer.length - position;

        if (remaining < 0)
        {
            return "";
        }
        else
        {
            int count = Math.min(length, remaining);
            String field = new String(buffer, position, count);
            position += count;

            // Util.unpadRight
            int i = field.length() - 1;

            while (i >= 0 && field.charAt(i) == ' ') i--;

            return field.substring(0, (i + 1));
        }
    }


    public String readVarchar(int length)
    {
        int end = getSmallint(0, length);

        String field = new String(get(length), 0, end);

        return Util.unpadRight(field, Util.SPACE);
    }


    public int readSmallint()
    {
        return getSmallint(Util.MIN_SMALLINT, Util.MAX_SMALLINT);
    }


    public int readInteger()
    {
        return getInteger(Util.MIN_INTEGER, Util.MAX_INTEGER);
    }


    public double readPic(boolean signed, int length, int fraction)
    {
        boolean negative = false;

        if (signed)
        {
            char sign = (char)get();

            if (sign == '-') negative = true;

            length--;
        }

        double field = decodeReal(length, fraction);

        if (negative && !Double.isNaN(field)) field *= -1;

        return field;
    }


    public double readDec(int length, int fraction)
    {
        return decodeReal((length + 1), fraction);
    }


    public java.sql.Date readDate()
    {
        int intDate = getInteger(0, Util.MAX_INTEGER);
        return TemporalFunctions.oldToNewDate(intDate);
    }


    public java.sql.Time readTime()
    {
        int intTime = getInteger(0, Util.MAX_TIME);
        return TemporalFunctions.oldToNewTime(intTime);
    }


    public java.sql.Timestamp readTimestamp()
    {
        java.sql.Date date = readDate();
        java.sql.Time time = readTime();
        int fraction = getInteger(Util.MIN_INTEGER, Util.MAX_INTEGER);

        return TemporalFunctions.timestamp(date, time, fraction);
    }


    // ------------------------------------------------------------------------


    private int getInteger(int min, int max)
    {
        byte[] word = get(4);

        int field = 0;

        for (int i = 3; i >= 0; i--)
            field = (field << 8) | (0xFF & word[i]);

        if (field < min) field = min;
        else if (field > max) field = max;

        return field;
    }


    private int getSmallint(int min, int max)
    {
        byte[] word = get(2);

        int field = (word[1] << 8) | (0xFF & word[0]);

        if (field < min) field = min;
        else if (field > max) field = max;

        return field;
    }


    private double decodeReal(int length, int fraction)
    {
        int digits = length - fraction;

        String integer = new String(get(digits));
        String number;
        
        if (fraction == 0) number = integer + ".0";
        else
        {
            String decimal = new String(get(fraction));

            number = integer + "." + decimal;
        }

        char c = number.charAt(0);

        number = Util.unpadLeft(number, (c == '0') ? Util.ZERO : Util.SPACE);

        try
        {
            return Double.valueOf(number).doubleValue();
        }
        catch (NumberFormatException e)
        {
            return Double.NaN;
        }
    }
}
