package com.ibm.ie.reeng.rt.common;


import java.io.PrintWriter;
import java.io.OutputStreamWriter;
import java.io.FileWriter;


/**
 * This is the default implementation of {@link Trace} which is used by
 * the {@link Log} class. It provides timestamped logging with the
 * option of redirecting the log entries to a file if the
 * &quot;trace.file&quot; system property is set. The default is to send
 * the log entries to the standard error output stream.
 */
public class FullTrace implements Trace
{
    private PrintWriter writer;
    private long start = System.currentTimeMillis();


    public FullTrace()
    {
        try
        {
            String file = Config.instance().getString("trace.file");

            if (file == null)
                writer = new PrintWriter(new OutputStreamWriter(System.out));
            else writer = new PrintWriter(new FileWriter(file));
        }
        catch (Exception e)
        {
            e.printStackTrace();
            System.exit(1);
        }
    }


    /**
     * Return the name of the given object's class.
     * <p>
     * If the object is a string it's simply returned - this allows
     * static functions to use the information, warning and error
     * functions.
     */
    private String name(Object source)
    {
        if (source instanceof String) return source.toString();
        return source.getClass().getName();
    }


    /**
     * We want to avoid allocating unnecessary Objects.
     */
    private void writeElapsedTime()
    {
        long diff = System.currentTimeMillis() - start;

        long centis = (diff / 10) % 100;
        long secs = (diff / 1000) % 60;
        long mins = (diff / 60000) % 60;
        long hours = diff / 3600000;

        if (hours < 10) writer.print("0");
        writer.print(hours);
        writer.print(":");

        if (mins < 10) writer.print("0");
        writer.print(mins);
        writer.print(":");

        if (secs < 10) writer.print("0");
        writer.print(secs);
        writer.print(":");

        if (centis < 10) writer.print("0");
        writer.print(centis);
    }


    /**
     * Flush the output stream and check for errors, exit if there are any.
     */
    private void flush()
    {
        if (writer.checkError()) // NB: checkError also does flush.
        {
            System.out.println("FullTrace: error during output.");
            System.exit(1);
        }
    }

    private void message(Object source, String kind, String text)
    {
        writeElapsedTime();
        writer.print(" ");
        writer.print(kind);
        writer.print(" - ");
        writer.print(name(source));
        writer.print(": ");
        writer.print(text);
        writer.println();
        flush();
    }

    public void output(String text)
    {
        writer.print(text);
        flush();
    }

    public void information(Object source, String text)
    {
        message(source, "Information", text);
    }

    public void warning(Object source, String text)
    {
        message(source, "Warning", text);
    }

    public void error(Object source, String text)
    {
        message(source, "Error", text);
    }

    public void exception(Exception e)
    {
        // A BailoutException is supposed to be equivalent to an exit
        // so unless it is explicitly caught and discared it should be
        // thrown all the way to the top of the stack.
        if (e instanceof BailoutException) throw (BailoutException)e;

        writeElapsedTime();
        writer.print(" ");
        e.printStackTrace(writer);
        flush();
    }

    public void fatalError(Object source, String text)
    {
        error(source, text);
        die(true);
    }

    public void fatalException(Exception e)
    {
        exception(e);
        die(false);
    }

    /**
     * Throw a runtime exception which will cause the application to
     * die.
     */
    private void die(boolean trace)
    {
        if (trace)
        {
            Throwable throwable = new Throwable();

            throwable.printStackTrace(writer);
            flush();
        }

        throw new BailoutException();
    }


    /**
     * Wired to self-destruct.
     */
    public static class BailoutException extends RuntimeException
    {
        public String getMessage()
        {
            System.out.println(
                "Log: abnormal termination see log for details");
            System.exit(1);

            return null;
        }
    }
}
