package com.ibm.ie.reeng.rt.common;

import java.util.Date;


/**
 * This is the standard {@link Marshall} extension for host
 * communications.  This is a character rather than binary based
 * encoding for data.
 */
public class DefaultMarshall extends StringMarshall
{
    public boolean readBoolean()
    {
        throw new RuntimeException("boolean marshalling not yet supported");
    }
    
    public Marshall writeBoolean(boolean field)
    {
        throw new RuntimeException("boolean marshalling not yet supported");
    }
    
    /**
     * Determines whether to log lots of marshalling debug information.
     * This is initialised by the system property "trace.marshall".
     */
    protected static boolean
        trace = Config.instance().getBoolean("trace.marshall");

    private final static int SMALLINT_DIGITS = 5;
    private final static int INTEGER_DIGITS = 10;

    /**
     * Writes a <code>String</code> to the character buffer. The string
     * is padded with <code>length</code> spaces or truncated as
     * necessary.
     */
    public Marshall writeChar(String field, int length)
    {
        put(Util.padRight(field, length, Util.SPACE));
        return this;
    }

    /**
     * Writes a <code>String</code> to the character buffer. The
     * actual length of the string is written first (see
     * {@link #putInteger}) followed by the string
     * contents.  The string is padded if necessary upto
     * <code>length</code> with spaces.
     */
    public Marshall writeVarchar(String field, int length)
    {
        putInteger(field.length(), SMALLINT_DIGITS, 0, Util.MAX_SMALLINT);
        put(Util.padRight(field, length, Util.SPACE));
        return this;
    }

    /**
     * Writes an <code>int</code> in a human readable form to the
     * character buffer. The value of the argument must be representable
     * by a signed 16 bit integer.
     */
    public Marshall writeSmallint(int field)
    {
        putInteger(field, SMALLINT_DIGITS,
            Util.MIN_SMALLINT, Util.MAX_SMALLINT);
        return this;
    }

    /**
     * Writes an <code>int</code> in a human readable form to the
     * character buffer. There is no restriction on the value of the
     * <code>int</code> to be written.
     */
    public Marshall writeInteger(int field)
    {
        putInteger(field, INTEGER_DIGITS,
            Util.MIN_INTEGER, Util.MAX_INTEGER);
        return this;
    }

    /**
     * Writes a <code>double</code> in a human readable form to the
     * character buffer. The value of the <code>double</code> must
     * conform to the other arguments (e.g. must be positive if
     * <code>signed</code> is <code>false</code>) otherwise truncation
     * will occur.
     */
    public Marshall writePic(double field, boolean signed,
        int length, int fraction)
    {
        if (signed) length--;
        writeDec(field, length, fraction);
        return this;
    }


    /**
     * Writes a <code>double</code> in a human readable form to the
     * character buffer. The value of the <code>double</code> must
     * conform to the <code>length</code> and <code>fraction</code>
     * arguments otherwise truncation will occur.
     */
    public Marshall writeDec(double field, int length, int fraction)
    {
        // Note: the field length of DEC data items does not include the sign.

        String string = Util.toDoubleString(field);
        char sign;

        if (string.charAt(0) == '-')
        {
            sign = '-';
            string = string.substring(1, string.length());
        }
        else sign = '+';

        int point = string.indexOf('.');
        String integer = string.substring(0, point);
        String decimal = string.substring((point + 1), string.length());

        integer = Util.padLeft(integer, (length - fraction), Util.ZERO);
        decimal = Util.padRight(decimal, fraction, Util.ZERO);

        string = sign + integer + decimal;

        if (trace) trace("actually sent " + string);

        put(string);
        return this;
    }


    /**
     * Writes a {@link java.sql.Date} to the character buffer. The date
     * is written as an integer which represents the number of days
     * since the Hps epoch (see {@link TemporalFunctions} for
     * discussion).
     */
    public Marshall writeDate(java.sql.Date field)
    {
        putInteger(TemporalFunctions.newToOldDate(field), INTEGER_DIGITS,
            Util.MIN_INTEGER, Util.MAX_INTEGER);
        return this;
    }


    /**
     * Writes a {@link java.sql.Time} to the character buffer. The date
     * is written as an integer which represents the number of
     * milliseconds since midnight (see {@link TemporalFunctions} for
     * discussion).
     */
    public Marshall writeTime(java.sql.Time field)
    {
        putInteger(TemporalFunctions.newToOldTime(field), INTEGER_DIGITS,
            Util.MIN_INTEGER, Util.MAX_INTEGER);
        return this;
    }


    /**
     * Writes a {@link java.sql.Timestamp} to the character buffer. The date
     * is written as three integers the first represents the date (see 
     * {@link #writeDate}, the second the time (see {@link #writeTime})
     * and the third a fraction representing a number of picoseconds.
     * See {@link TemporalFunctions} for a discussion on the temporal
     * type mapping.
     */
    public Marshall writeTimestamp(java.sql.Timestamp field)
    {
        int date = TemporalFunctions.newToOldDate(
            TemporalFunctions.date(field));
        int time = TemporalFunctions.newToOldTime(
            TemporalFunctions.time(field));
        int fraction = TemporalFunctions.fraction(field);

        putInteger(date, INTEGER_DIGITS,
            Util.MIN_INTEGER, Util.MAX_INTEGER);
        putInteger(time, INTEGER_DIGITS,
            Util.MIN_INTEGER, Util.MAX_INTEGER);
        putInteger(fraction, INTEGER_DIGITS,
            Util.MIN_INTEGER, Util.MAX_INTEGER);
        return this;
    }


    // ------------------------------------------------------------------------


    /**
     * A utility method for writing arbitrary <code>int</code>s in a
     * character (human readable) form.
     */
    protected void putInteger(int field, int digits, int min, int max)
    {
        if (field < min) field = min;
        else if (field > max) field = max;

        if (field < 0)
        {
            put("-");
            field *= -1;
        }
        else put("+");

        put(Util.padLeft(Integer.toString(field), digits, Util.ZERO));
    }


    // ------------------------------------------------------------------------


    /**
     * Reads <code>length</code> characters from the internal buffer and
     * returns the result <code>String</code> with spaces trimmed from
     * the right.
     */
    public String readChar(int length)
    {
        if (trace) trace("parsing a CHAR(" + length + ")");
        return Util.unpadRight(get(length), Util.SPACE);
    }


    /**
     * Returns a <code>String</code> representing an encoded VARCHAR at
     * the current position in the internal buffer. See {@link
     * #writeVarchar} for a note on the encoding.
     */
    public String readVarchar(int maxLength)
    {
        if (trace) trace("parsing a VARCHAR(" + maxLength + ")");

        int length = getInteger(SMALLINT_DIGITS);
        String string = get(maxLength);

        if (maxLength > length) string = string.substring(0, length);

        return string;
    }

    /**
     * Reads a character representation of a 16 bit signed integer from
     * the internal character buffer and returns a corresponding
     * <code>int</code>.
     */
    public int readSmallint()
    {
        return getInteger(SMALLINT_DIGITS);
    }

    /**
     * Reads a character representation of a 32 bit signed integer from
     * the internal character buffer and returns a corresponding
     * <code>int</code>.
     */
    public int readInteger()
    {
        return getInteger(INTEGER_DIGITS);
    }


    public double readPic(boolean signed, int length, int fraction)
    {
        if (signed) length--;
        return readDec(length, fraction);
    }


    public double readDec(int length, int fraction)
    {
        // Note: the field length of DEC data items does not include the sign.

        String string = get(length + 1);
        boolean minus = (string.charAt(0) == '-');

        string = string.substring(1, string.length()); // Remove sign byte.

        if (trace) trace("got DEC(" + length + ", " + fraction
            + ") string \"" + string + "\"");

        if (fraction > 0)
        {
            if (fraction == length) string = "0." + string;
            else
            {
                String integer = string.substring(0, (length - fraction));

                String decimal = string.substring((length - fraction),
                    string.length());

                string = integer + "." + decimal;
            }
        }

        double field = parseDouble(string);

        if (minus) field *= -1;

        return field;
    }


    public java.sql.Date readDate()
    {
        return TemporalFunctions.oldToNewDate(getInteger(INTEGER_DIGITS));
    }


    public java.sql.Time readTime()
    {
        return TemporalFunctions.oldToNewTime(getInteger(INTEGER_DIGITS));
    }


    public java.sql.Timestamp readTimestamp()
    {
        java.sql.Date date = readDate();
        java.sql.Time time = readTime();
        int fraction = getInteger(INTEGER_DIGITS);

        return TemporalFunctions.timestamp(date, time, fraction);
    }


    // ------------------------------------------------------------------------


    private int getInteger(int digits)
    {
        String string = get(digits + 1);

        if (trace) trace("trying to parse \"" + string + "\" as an int");

        boolean negative = (string.charAt(0) == '-');

        string = string.substring(1, string.length());
        string = Util.unpadLeft(string, Util.ZERO);

        if (string.length() == 0) return 0;

        int field = Integer.parseInt(string);

        if (negative) field *= -1;

        return field;
    }

    private double parseDouble(String string)
    {
        string = Util.unpadLeft(string, Util.ZERO);

        if (trace) trace("trying to parse \"" + string + "\" as a double");

        if (string.length() == 0) return 0;

        return Double.valueOf(string).doubleValue();
    }

    public void viewEnd()
    {
        if (trace) trace("finished marshalling view. length is " + getLength());
    }

    private void trace(String message)
    {
        Log.information(this, message);
    }
}
