package com.ibm.ie.reeng.rt.common;

import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;

/**
 * This class supports the complex assignement statements in Hps.
 */
public class Assign
{
    private Assign() { }

    /**
     * Maps a source field, view or array to a compatable destination.
     * <p>
     * In the case of fields, all field types are immutable and thus
     * cannot be directly changed. Therefore this method returns the new
     * destination <code>Object</code>. In the case of views and arrays
     * the returned <code>Object</code> will be the same as the
     * <code>dest</code> parameter; however the destination will have
     * been updated.
     * <p>
     * Mapping a view to a view in Hps involves recursively mapping
     * eponymous fields from the source view to the destination. Mapping
     * fields involves simple assignment. Mapping arrays involves
     * recursively mapping as many as possible of the elements from the
     * source to the destination.
     */
    public static Object byName(Object src, Object dest)
    {
        try
        {
            if (src == null || dest == null)
            {
                Log.warning("Assign", "null object passed to byName().");
                return null;
            }
            Class srcClass = src.getClass();
            Class destClass = dest.getClass();

            if (isBasicType(srcClass) && isBasicType(destClass))
                return src;

            if (srcClass.isPrimitive() || destClass.isPrimitive())
                Log.fatalError("Assign", "can not map from " +
                    srcClass.getName() + " to " + destClass.getName());

            if (srcClass.isArray() && destClass.isArray())
            {
                int length =
                    Math.min(Array.getLength(src), Array.getLength(dest));

                for (int i = 0; i < length; i++)
                    Array.set(dest, i,
                        byName(Array.get(src, i), Array.get(dest, i)));
            }
            else if (srcClass.isArray())
            {
                return byName(Array.get(src, 0), dest);
            }
            else if (destClass.isArray())
            {
                Array.set(dest, 0, byName(src, Array.get(dest, 0)));
            }
            else
            {
                Field[] srcFields = srcClass.getFields();
                Field[] destFields = destClass.getFields();

                boolean assignmentHappened = false;

                for (int i = 0; i < srcFields.length; i++)
                {
                    Field srcField = srcFields[i];
                    String srcName = srcField.getName();

                    for (int j = 0; j < destFields.length; j++)
                    {
                        Field destField = destFields[j];

                        if (!Modifier.isFinal(destField.getModifiers())
                            && destField.getName().equals(srcName))
                        {
                            destField.set(dest,
                                byName(srcField.get(src), destField.get(dest)));
                            assignmentHappened = true;
                            break;
                        }
                    }
                }

                if (!assignmentHappened)
                {
                    Log.fatalError("Assign", "can not map from " +
                        srcClass.getName() + " to " + destClass.getName());
                }
            }
        }
        catch (Exception e)
        {
            Log.fatalException(e);
        }

        return dest;
    }

    public static void byType(View source, View dest)
    {
        try
        {
            Field[] srcFields = source.getClass().getFields();
            Field[] destFields = dest.getClass().getFields();

            boolean assignmentHappened = false;
            
            int len = Math.min(srcFields.length, destFields.length);

            for (int i = 0; i < len; i++)
            {
                Class srcType = srcFields[i].getType();
                Class destType = destFields[i].getType();

                if (isBasicType(srcType) && srcType.equals(destType))
                {
                    destFields[i].set(dest, srcFields[i].get(source));
                }
                else if (isNumeric(srcType) && isNumeric(destType))
                {
                    Number n = (Number)srcFields[i].get(source);

                    if (destType.equals(Integer.class))
                        destFields[i].setInt(dest, n.intValue());
                    else if (destType.equals(Long.class))
                        destFields[i].setLong(dest, n.longValue());
                    else destFields[i].setDouble(dest, n.doubleValue());
                }
                else
                {
                    Object srcChild = srcFields[i].get(source);
                    Object destChild = destFields[i].get(dest);
                    
                    if (srcChild instanceof View && destChild instanceof View)
                        byType((View)srcChild, (View) destChild);
                    else break;
                }
            }
        }
        catch (Exception e)
        {
            Log.fatalException(e);
        }
    }

    private static boolean isNumeric(Class c)
    {
        return c.equals(Short.class)
            || c.equals(Integer.class)
            || c.equals(Long.class)
            || c.equals(Double.class);
    }

    /**
     * Note the (strictly speaking incorrect) assumption that the
     * java.sql temporal classes are immutable.
     */
    private static boolean isBasicType(Class type)
    {
        return
            (type == Integer.class ||
             type == Long.class  ||
             type == Double.class  ||
             type == String.class  ||
             type == java.sql.Date.class ||
             type == java.sql.Time.class ||
             type == java.sql.Timestamp.class);
    }
}
