package com.ibm.ie.reeng.rt.client;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.beans.Beans;
import java.util.Iterator;
import java.util.StringTokenizer;

import javax.swing.AbstractAction;
import javax.swing.DefaultBoundedRangeModel;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JScrollBar;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JViewport;
import javax.swing.KeyStroke;
import javax.swing.ListCellRenderer;
import javax.swing.ListSelectionModel;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.border.EmptyBorder;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;

import com.ibm.ie.reeng.rt.common.Config;



/**
 * This class implements functionality equivalent to Hps SPREADSHEETs
 * using a JTable.
 * <p>
 * SPREADSHEETs have an attribute AUTOCALL, it is false by default.  If
 * AUTOCALL is set to true HPS_LB_TOP and HPS_LB_BOTTOM events are
 * generated when the user attempts to scroll beyond the top or bottom
 * of the table.
 * <p>
 * Note that contrary to other sources calling the system components
 * HPS_TBL_INIT_SIZE or SET_VIRTUAL_LISTBOX_SIZE in no way effects the
 * setting of AUTOCALL or the behavior it causes.
 * <p>
 * HPS_LB_TOP and HPS_LB_BOTTOM events are not generated on simply
 * scrolling to the top or bottom of the SPREADSHEET. They are only
 * generated on attempting to scroll beyond the top or bottom by
 * pressing the up or down scrollbar arrows on the vertical scrollbar or
 * by navigating up or down using the cursor keys.  Pressing the top or
 * bottom scrollbar arrows or the up or down cursor keys multiple times
 * when at the top or the bottom of the SPREADSHEET will generate
 * multiple HPS_LB_TOP and HPS_LB_BOTTOM events.
 * Note: trying to drag the scrollbar thumb/elevator up or down beyond
 * the top or bottom of the SPREADSHEET does not cause HPS_LB_TOP and
 * HPS_LB_BOTTOM events.
 * <p>
 * If the system components HPS_TBL_INIT_SIZE or
 * SET_VIRTUAL_LISTBOX_SIZE are called then the SPREADSHEET will
 * generate HPS_LB_OUTRANGE events when the datalinked view
 * providing its data does not contain an item of data the table
 * needs to display (the user having scrolled up or down beyond the
 * current contents of the view).
 * <p>
 * Each time the system component HPS_TBL_INIT_SIZE is called on a
 * SPREADSHEET the SPREADSHEET will generate a once off INIT event
 * when the window containing it is next conversed (causing the
 * converse to return immediately).  While the system component
 * SET_VIRTUAL_LISTBOX_SIZE is almost identical to HPS_TBL_INIT_SIZE
 * calling it will not result in an INIT event being generated.
 * <p>
 * If AUTOCALL is not set and neither the system component
 * HPS_TBL_INIT_SIZE nor SET_VIRTUAL_LISTBOX_SIZE are called a
 * SPREADSHEET will generate no events whatsoever.
 * <p>
 * While the system component HPS_TBL_INIT_SIZE supercedes
 * SET_VIRTUAL_LISTBOX_SIZE there is one thing SET_VIRTUAL_LISTBOX_SIZE
 * can do one that HPS_TBL_INIT_SIZE can't - set the thumb/elevator
 * position of a SPREADSHEET's vertical scroll bar.  Note: setting the
 * elevator position with SET_VIRTUAL_LISTBOX_SIZE to zero doesn't mean
 * leave it alone - it means just the same as setting it to one.
 * <p>
 * Calling HPS_TBL_INIT_SIZE causes the thumb/elevator of the
 * SPREADSHEET's vertical scrollbar to be reset back up to the top.
 */
public class TransmuteTable extends JTable
    implements ChangeListener, FocusListener, Identifier, DataHandler.Interface, 
               java.io.Serializable
{
    private boolean designTime=false;
    
    // table header padding, different UIs need different values
    private int padding = 0;
    
    public TransmuteTable()
    {
    	padding = Config.instance().getInt("table.header.padding");
    	padding = ((padding==0)?0:padding);
    	
        designTime=Beans.isDesignTime();

        if (!designTime)
        {
            setAutoResizeMode(JTable.AUTO_RESIZE_OFF);

            addFocusListener(this);

            // Set single row selection as the default.
            setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

            // Replace up and down cursor key actions in order to be
            // able to generate HPS_LB_TOP and HPS_LB_BOTTOM events.

            replaceKeyboardAction(KeyStroke.getKeyStroke(KeyEvent.VK_UP, 0));
            replaceKeyboardAction(KeyStroke.getKeyStroke(KeyEvent.VK_UP,
                ActionEvent.SHIFT_MASK));

            replaceKeyboardAction(KeyStroke.getKeyStroke(KeyEvent.VK_DOWN, 0));
            replaceKeyboardAction(KeyStroke.getKeyStroke(KeyEvent.VK_DOWN,
                ActionEvent.SHIFT_MASK));

            // so that error cells are rendered with a red background.
            setDefaultRenderer(Object.class, new CellRenderer());
        }
    }


    // ------------------------------------------------------------------------


    //
    // TODO: add appropriate calls to this method.
    //
    public void resetState()
    {
        if (!designTime)
        {
            previousY = 0;
            currentTopRow = 0;

            // Scroll back to origin - without firing any events.
            model.enableEvents(false);
            getScrollPane().getVerticalScrollBar().setValue(0);
            getScrollPane().getHorizontalScrollBar().setValue(0);
            model.enableEvents(true);

            model.resetState();
        }
    }


    // ------------------------------------------------------------------------

    // These methods provide an iterator type interface to the selected
    // rows of the table. These methods should only be used by the
    // GetSelectedField system component as it depends on the table
    // maintaining the iterator state.

    private int[] selectedRows;
    private int selectedIndex;

    public void storeSelectedRows()
    {
        selectedRows = getSelectedRows();
        selectedIndex = 0;
    }

    public boolean hasNextSelectedRow()
    {
        return selectedIndex < selectedRows.length;
    }

    public int nextSelectedRow()
    {
        return selectedRows[selectedIndex++];
    }

    public void setMandatory(boolean mandatory)
    {
        getDataHandler().setMandatory(mandatory);
    }

    public boolean getMandatory()
    {
        return getDataHandler().getMandatory();
    }

    public void setImmediateReturn(boolean immediateReturn)
    {
        getDataHandler().setImmediateReturn(immediateReturn);
    }

    public boolean getImmediateReturn()
    {
        return getDataHandler().getImmediateReturn();
    }

    // ------------------------------------------------------------------------


    //
    // If the table gains focus...
    //
    // a. if there are any cells for which there are not available
    //    values scroll them out of site.
    //
    // b. if one of the table's cells contains a bad value make it the
    //    selected cell.
    //
    public void focusGained(FocusEvent e)
    {
        if (!designTime)
        {
            int[] availableRows = model.getAvailableRows();
            int availableTop = availableRows[0];
            int availableBottom = availableRows[1];

            int[] visibleRows = getVisibleRows();
            int visibleTop = visibleRows[0];
            int visibleBottom = visibleRows[1];

            int rowHeight = getRowHeight() + getRowMargin();

            int value = -1;

            if (visibleTop < availableTop) value = availableTop * rowHeight;
            else if (visibleBottom > availableBottom)
                value = (availableBottom + 1) * rowHeight - getVisibleRect().height;

            if (value != -1)
            {
                if (value < 0) value = 0;
                else
                {
                    int bottom = getRowCount() * rowHeight - getVisibleRect().height;

                    if (value > bottom) value = bottom;
                }

                scrollBar.setValue(value);
            }

            // ------------------------------------------------------------

            int[] cell = model.getFirstErrorCell();

            if (cell != null)
            {
                int row = cell[0];
                int col = cell[1];

                Rectangle cellRect = getCellRect(row, col, false);

                scrollRectToVisible(cellRect);

                // You might imagine that calling editCellAt() would
                // be more appropriate than just selecting the cell
                // as follows - but you'd be very badly wrong.

                ListSelectionModel rsm = getSelectionModel();
                ListSelectionModel csm = getColumnModel().getSelectionModel();

                rsm.setSelectionInterval(row, row);
                csm.setSelectionInterval(col, col);
            }

    // This is experimental change to address an issue raised by CIC. The
    // reason it is here is that when a window appears with the focus in a
    // table, the row containing the table cursor is not selected. However
    // if you tab into a table, the row is selected.
    if (!getCellSelectionEnabled()
        && getSelectedRowCount() == 0)
    {
        setRowSelectionInterval(0, 0);
    }
        }
    }


    public void focusLost(FocusEvent e) { }


    // ------------------------------------------------------------------------


    private TableDataHandler handler = new TableDataHandler(this);


    public DataHandler getDataHandler() { return handler; }


    public void setClientData(ClientData clientData)
    {
        // TransmuteColumn manages data for the table.

        //1.2KL throw new UnsupportedOperationException();
        throw new RuntimeException("Unsupported operation: setClientData");
    }


    // ------------------------------------------------------------------------


    private TransmuteTableModel model;


    public TransmuteTableModel getTransmuteTableModel() { return model; }


    /**
     * Once the model has been set up this function is called. All it
     * does is add header renderers to each of the columns and sets up
     * the width for each of the columns.
     */
    public void setModel(final TransmuteTableModel model)
    {
        if (!designTime)
        {
            super.setModel(model);

            this.model = model;

            HeaderRenderer renderer = new HeaderRenderer();
            
            // carries a count of the space needed to resize the table
            int spaceNeeded = 0;                        
			// the number of columns in the table
			final int columnCount = getColumnCount();
            // retains the index of the most spacious column
            java.util.Vector spaciousColumnsIndex = new java.util.Vector(columnCount);
            java.util.Vector spaciousColumnsSize = new java.util.Vector(columnCount);          
            
            // the available space available
            int spaceAvailable = 0;
            
            TableColumn column = null;
			// 1.  Set the minimum width to be the greater of the column header width
			// and preferred width
            for (int i = 0; i < columnCount ; i++)
            {
                column = getColumnModel().getColumn(i);

                column.setHeaderRenderer(renderer);

                java.awt.Component component = renderer.
                    getTableCellRendererComponent(null, column.getHeaderValue(), 
                        false, false, 0, 0);

				// get the HPS preferred width plus a padding
                // runtime.properties				
                int width = model.getColumn(i).getPreferredWidth()+padding;
                
                // get the components min width plus a padding from the 
                // runtime.properties
                final int minWidth = component.getMinimumSize().width+padding;                                         
                
                // If no preferred width has been set for the column use
                // the width of the column header.				
				if(width==0)
				{
					width = component.getPreferredSize().width;
				}	
				
				// if the column width is too small for the text
				// set the width to fit and increase the space needed
				if(minWidth > width)
				{
					spaceNeeded +=(minWidth-width);
					width = minWidth;							
				}
				// if the width is large enough, record the space available and the index of the column
				else if(width > minWidth)
				{
					spaciousColumnsIndex.add(new Integer(i));
					spaciousColumnsSize.add(new Integer(width - minWidth));
					spaceAvailable+= width - minWidth;
				}
				column.setWidth(width);
                column.setPreferredWidth(width);
//                column.setMinWidth(width);
//                column.setMaxWidth(width);
            }
            // 2. Cycle through the available spacious columns and 
            // grab the space from them
            final Iterator indexIterator = spaciousColumnsIndex.iterator();
            final Iterator columnIterator = spaciousColumnsSize.iterator();
                        
            int columnSpace = 0;
            int columnIndex = 0;
            int columnReduction = 0;
            int newColumnWidth = 0;
            int remainingSpaceNeed = spaceNeeded;
            
            while(indexIterator.hasNext())
            {
           		// get the index of the spacious column
           		columnIndex = ((Integer)indexIterator.next()).intValue();            	
            	// get how much space the column has
           		columnSpace = ((Integer)columnIterator.next()).intValue();
           		// if its the last column => make the column reduction the remaining space needed 
           		if(!indexIterator.hasNext())
           		{
           			columnReduction = remainingSpaceNeed;
           		}
           		// else weight it verses the space needed and available and get the
           		// value to reduce the column width by           		
           		else
           		{
           			columnReduction = (int)(((double)columnSpace/spaceAvailable)*spaceNeeded);
	           		remainingSpaceNeed-= columnReduction;
           		}
           		// set the approprate column in the model to a new width           		
           		column = getColumnModel().getColumn(columnIndex);     
           		newColumnWidth = column.getWidth()-columnReduction ;
           		column.setWidth(newColumnWidth);
           		column.setPreferredWidth(newColumnWidth);
            }
            
            
            // if we still need space, remove it from the most spacious column
//            if(spaceNeeded > 0)
//            {
//            	TableColumn column = getColumnModel().getColumn(mostSpaciousColumnIndex);
//            	column.setWidth(column.getWidth()-spaceNeeded);            	
//            }

            // listen for double-clicks changes so we can implement
            // immediate returns.
            TransmuteColumn immediate = model.getFirstImmediateReturnColumn();
            if (immediate != null)
            {
                // The event source for doubleclicked table rows is
                // unusual compared to the other events in Hps:
                final String windowRetcode
                    = getIdentifier() + "." + immediate.getIdentifier();
                addMouseListener(new MouseAdapter()
                {
                    public void mouseClicked(MouseEvent e)
                    {
                        if (e.getClickCount() == 2)
                        {
                            TransmuteDialog.getParentDialog(TransmuteTable.this).
                                dispatchImmediateReturnEvent(windowRetcode);
                        }
                    }
                });
            }
        }
    }

    private TransmuteColumn getTransmuteColumn(int i)
    {
        return model.getColumn(i);
    }

    public TransmuteColumn getColumnById(String hpsId)
    {
        for (int i = 0; i < getColumnCount(); i++)
        {
            TransmuteColumn column = getTransmuteColumn(i);
            if (column.getIdentifier().equals(hpsId))
                return column;
        }
        return null;
    }

    public TransmuteColumn getColumnByDatalink(String view, String field)
    {
        for (int i = 0; i < getColumnCount(); i++)
        {
            TransmuteColumn column = getTransmuteColumn(i);
            if (column.matchLongName(view, field)) return column;
        }
        return null;
    }

    // ------------------------------------------------------------------------


    private boolean autoCall = false;


    public void setAutoCall(boolean autoCall) { this.autoCall = autoCall; }
    public boolean getAutoCall() { return autoCall; }


    /**
     * Replaces the keyboard actions for the up and down cursor keys so
     * that HPS_LB_TOP and HPS_LB_BOTTOM events can be generated. The
     * original actions are called by the new actions once they've done
     * their work.
     */
    private void replaceKeyboardAction(KeyStroke keyStroke)
    {
        NavigationAction action = new NavigationAction(this, keyStroke);

        registerKeyboardAction(action, keyStroke, JComponent.WHEN_FOCUSED);
    }


    private static class NavigationAction extends AbstractAction
    {
        private TransmuteTable table;
        private ActionListener originalAction;
        private boolean down;


        public NavigationAction(TransmuteTable table, KeyStroke keyStroke)
        {
            this.table = table;

            down = keyStroke.getKeyCode() == KeyEvent.VK_DOWN;

            originalAction = table.getActionForKeyStroke(keyStroke);
        }


        public void actionPerformed(ActionEvent event)
        {
            if (table.autoCall)
            {
                ListSelectionModel model = table.getSelectionModel();
                int leadRow = model.getLeadSelectionIndex();

                if (down)
                {
                    int bottom = table.getRowCount() - 1;

                    if (leadRow == bottom)
                        dispatch(TransmuteDialog.TABLE_EVENT_BOTTOM);
                }
                else
                {
                    if (leadRow == 0)
                        dispatch(TransmuteDialog.TABLE_EVENT_TOP);
                }
            }

            originalAction.actionPerformed(event);
        }


        private void dispatch(String name)
        {
            TransmuteDialog.getParentDialog(table).
                dispatchTableAutoCallEvent(table, name);
        }
    }


    /**
     * JScrollPane is extended so that the model for the vertical scroll
     * bar can be replaced at its point of creation (replacing it at a
     * latter point given the current implementation of JScrollPane
     * results in unexpected side effects). The new model provides the
     * ability to generate HPS_LB_TOP and HPS_LB_BOTTOM events.
     */
    public static class ScrollPane extends JScrollPane
    {
        private TransmuteTable table;


        public ScrollPane(TransmuteTable table)
        {
            super(table);

            this.table = table;
        }


        public JScrollBar createVerticalScrollBar()
        {
            JScrollBar scrollBar = super.createVerticalScrollBar();

            replaceScrollBarModel(scrollBar);

            return scrollBar;
        }


        private void replaceScrollBarModel(JScrollBar scrollBar)
        {
            DefaultBoundedRangeModel model =
                (DefaultBoundedRangeModel)scrollBar.getModel();

            int value = model.getValue();
            int extent = model.getExtent();
            int min = model.getMinimum();
            int max = model.getMaximum();

            model = new DefaultBoundedRangeModel(value, extent, min, max)
            {
                public void setValue(int val)
                {
                    super.setValue(val);

                    if (table.autoCall)
                    {
                        if (val < getMinimum())
                            dispatch(TransmuteDialog.TABLE_EVENT_TOP);
                        else if (val > (getMaximum() - getExtent()))
                            dispatch(TransmuteDialog.TABLE_EVENT_BOTTOM);
                    }
                }

                public String toString()
                {
                    return "MY SCROLLBAR MODEL: " + super.toString();
                }
            };

            scrollBar.setModel(model);
        }


        private void dispatch(String name)
        {
            TransmuteDialog.getParentDialog(table).
                dispatchTableAutoCallEvent(table, name);
        }
    }


    // ------------------------------------------------------------------------


    public void dispatchInitEvent()
    {
        TransmuteDialog.getParentDialog(this).
            dispatchTableOutRangeEvent(this, TransmuteDialog.TABLE_EVENT_INIT);
    }

    // ------------------------------------------------------------------------


    private JScrollBar scrollBar;
    private int previousY = 0;
    private int currentTopRow = 0;


    public int getFirstVisibleRowVirtualOccurrence()
    {
        return currentTopRow;
    }

    public int getElevatorPosition() { return model.getElevatorPosition(); }


    public int getVisibleRowCount()
    {
        int rowHeight = getRowHeight() + getRowMargin();

        return (int)Math.ceil(
            (double)getVisibleRect().height / (double)rowHeight);
    }

    public void setElevatorPosition(int newValue)
    {
        model.setElevatorPosition(newValue);
    }

    public void setFirstVisibleRow(int virtualRowNo)
    {
        model.enableEvents(false);
        JViewport vp = (JViewport)getParent();
        Point p = new Point(0, (getRowHeight() + getRowMargin()) * virtualRowNo);
        vp.setViewPosition(p);
        model.enableEvents(true);
    }

    public void addNotify()
    {
        super.addNotify();

        scrollBar = getScrollPane().getVerticalScrollBar();

        getViewport().addChangeListener(this);
    }

    public void stateChanged(ChangeEvent event)
    {
        TransmuteDialog dialog = TransmuteDialog.getParentDialog(this);
        if (dialog.isVisible()) dialog.waitForConverse();

        int value = getVisibleRect().y;

        // Note: if you called getValue() on the the scroll pane's
        // scroll bars they would return an incorrect value at this
        // point in time. There values are updated only after this
        // stateChanged() method is called.

        if (previousY == value) return;

        boolean down = value > previousY;

        previousY = value;

        int[] visibleRows = getVisibleRows();

        int topRow = visibleRows[0];
        int bottomRow = visibleRows[1];

        currentTopRow = topRow;

        model.visibleRangeChanged(down, topRow, bottomRow);
    }


    private int[] getVisibleRows()
    {
        Rectangle visible = getVisibleRect();

        int rowHeight = getRowHeight() + getRowMargin();
        int topRow = visible.y / rowHeight;
        int bottomRow = (int)Math.ceil(
            (double)(visible.y + visible.height) / (double)rowHeight) - 1;

        return new int[]{ topRow, bottomRow };
    }


    private JScrollPane getScrollPane()
    {
        JViewport viewport = getViewport();

        if (viewport != null)
        {
            java.awt.Component parent = viewport.getParent();

            if (parent != null && parent instanceof JScrollPane)
                return (JScrollPane)parent;
        }

        return null;
    }


    // ------------------------------------------------------------------------


    private String identifier;

    public String getIdentifier() { return identifier; }

    public void setIdentifier(String identifier)
    {
        this.identifier = identifier;
    }


    // ------------------------------------------------------------------------


    /**
     * Sets the focus to a specific row and column.
     */
    public void requestFocus(int row, int col)
    {
        requestFocus();

        if (row >= getRowCount()) return;
        if (col >= getColumnCount()) return;

        Rectangle rectangle = getCellRect(row, col, true);

        scrollRectToVisible(rectangle);

        setRowSelectionInterval(row, row);
        setColumnSelectionInterval(col, col);
    }


    public boolean isFocusTraversable()
    {
        return (getRowCount() > 0 ? super.isFocusTraversable() : false);
    }


    // ------------------------------------------------------------------------


    /**
     * This function returns the preferred size of the table with its
     * width adjusted to be at least as wide as its scroll pane parent.
     */
    public Dimension getPreferredSize()
    {
        int width = getViewport().getExtentSize().width;

        Dimension d = super.getPreferredSize();

        if (d.width < width) d.width = width;

        return d;
    }


    private JViewport getViewport()
    {
        java.awt.Component parent = getParent();

        if (parent != null && parent instanceof JViewport)
            return (JViewport)parent;

        return null;
    }


    // ------------------------------------------------------------------------

	/**
	 * GJ: 20031216 -  changed to implement multi-line headers
	 */
    private static class HeaderRenderer
        extends JList implements TableCellRenderer
    {
        public HeaderRenderer()
        {
            setOpaque(true);
            setBorder(UIManager.getBorder("TableHeader.cellBorder"));
			ListCellRenderer renderer = getCellRenderer();
			((JLabel)renderer).setHorizontalAlignment(JLabel.LEFT);
			setCellRenderer(renderer);       
            //setEditable(false);
        }


        public void updateUI()
        {
            super.updateUI(); 
            setForeground(null);
            setBackground(null);
        }

		public java.awt.Component getTableCellRendererComponent(JTable table, Object value,
			       boolean isSelected, boolean hasFocus, int row, int column) 
		{
			if(table!=null)
			{
				setFont(table.getFont());
			}
			final String str = (value == null) ? "" : value.toString();
			
			// have a string in the format "firstline\nsecond line\n etc.."
			StringTokenizer st = new StringTokenizer(str,"\n");
			// set the vectors initial size to 4
			java.util.Vector header = new java.util.Vector(4);
		
			while(st.hasMoreElements())
			{
				header.add(st.nextToken());
			}
			setListData(header);		
			return this;				
		}		
    }

    private static class CellRenderer
        extends JLabel implements TableCellRenderer
    {
        private static Border noFocusBorder = new EmptyBorder(1, 2, 1, 2);
        public CellRenderer()
        {
            setOpaque(true);
            setBorder(noFocusBorder);
            setHorizontalAlignment(JLabel.CENTER);
        }

        public java.awt.Component getTableCellRendererComponent(
            JTable table,
            Object value,
            boolean isSelected,
            boolean hasFocus,
            int row,
            int column)
        {
            Color background = null;

            TransmuteTableModel model
                = ((TransmuteTable)table).getTransmuteTableModel();

            boolean error = model.isErrorCell(row, column);
            int alignment = model.getColumn(column).getHorizontalAlignment();

            setHorizontalAlignment(alignment);

            if (error) background = Color.red;

            if (isSelected)
            {
                setForeground(table.getSelectionForeground());
                setBackground(table.getSelectionBackground());
            }
            else
            {
                setForeground(table.getForeground());
                setBackground((background != null) ? background
                    : table.getBackground());
            }

            if (hasFocus)
            {
                setBorder(UIManager.getBorder("Table.focusCellHighlightBorder"));

                if (table.isCellEditable(row, column))
                {
                    setForeground(UIManager.getColor("Table.focusCellForeground"));
                    setBackground(UIManager.getColor("Table.focusCellBackground"));
                }
            }
            else
            {
                setBorder(noFocusBorder);
            }
            setValue(value);
            return this;
        }


        public void updateUI()
        {
            super.updateUI(); 
            setForeground(null);
            setBackground(null);
        }

        protected void setValue(Object value)
        {
            setText((value == null) ? "" : value.toString());
        }
    }
}
