package com.ibm.ie.reeng.rt.client;

import com.ibm.ie.reeng.rt.common.*;

import java.net.*;
import java.io.*;

/**
 * This is the base class for "proxy rules" which will connect
 * to a Servlet. As the name would suggest, subclasses act as 
 * proxies for remote rules.
 * <p>
 * This class depends on a number of runtime properties:
 * <UL>
 * <LI>"servlet.url" - URL of the Servlet.
 * <LI>"trace.requests" - if true log all requests sent to the server.
 * <LI>"trace.responses" - if true log all responses received from the server.
 * <LI>"trace.stack" - write a stack trace to the log every time this
 *      rule is used.
 * </UL>
 *
 */
public abstract class ServletProxyRule extends Rule
{
    private static String servletUrl;
    private static boolean dumpRequest;
    private static boolean dumpResponse;
    private static boolean dumpStack;

    // NB: Should we open a connection each time?
    //     I would imagine so
    private URL url;

    /**
     * Initialize all the statics here.
     */
    static
    {
        // get the debug settings.
        servletUrl=Config.instance().getString("servlet.url");
        dumpRequest = Config.instance().getBoolean("trace.requests");
        dumpResponse = Config.instance().getBoolean("trace.responses");
        dumpStack = Config.instance().getBoolean("trace.stack");
    }

    private static void die(String message)
    {
        Log.fatalError("ServletProxyRule", message);
    }

    protected ServletProxyRule()
    {
        try
        {
            url=new URL(servletUrl);
        }
        catch (MalformedURLException e)
        {
            Log.exception(e);
        }
    }

    /** Subclasses override this to return the input view. */
    public View getInputView() { return null; }

    public void clearInputView() { }

    /** 
     * Subclasses override this to return the server response to
     * the output view.
     */
    public View getOutputView() { return null; }

    public void setOutputView(View out) {}
    
    public void setInputView(View out) {}

    /** 
     * Subclasses implement this to specify the name of the remote rule
     * to invoke. Normally this will return the Hps shortname of the
     * rule.
     */
    protected abstract String getRemoteName();

    public void invokeRemote()
    {
        Log.information(this, "entering proxy rule " + getRemoteName());

        if (dumpStack) Location.logStack();

        try
        {
			HttpURLConnection httpCon=
				(HttpURLConnection)url.openConnection();
			httpCon.setDoOutput(true);
			httpCon.setRequestMethod("POST");
			
			ObjectOutputStream oos=
				new ObjectOutputStream(httpCon.getOutputStream());

//            if (dumpRequest) marshall.dumpBuffer();
            
            // Send the rule name
            oos.writeObject(getRemoteName());

            // If there's an input view, send it
            if (getInputView() != null) 
                oos.writeObject(getInputView());

            oos.close();

            // int len = getOutputView().writeTo(new SizeofMarshall()).getLength();

//            comms.send(getRemoteName(), marshall);

//            if (dumpResponse) marshall.dumpBuffer();

            // If there's an output view, retrieve it
            
            ObjectInputStream ois=
                new ObjectInputStream(httpCon.getInputStream());

			String result=(String)ois.readObject();
			
			if (result.equals("Error"))
			{
				System.out.println("Received an Error:");
				System.out.println(ois.readObject());
			}
			else if (result.equals("Response"))
			{
				System.out.println("Success:");
                if (getOutputView() != null) 
                {
                    setOutputView((View)ois.readObject());
                }
			}
			else
			{
				System.out.println("Bad response from Server");
			}
				
			ois.close();
            
        }
        catch (Exception e)
        {
            Log.exception(e);
        }

        Log.information(this, "exiting proxy rule " + getRemoteName());
    }

    public String toString()
    {
        return Util.toString(this, 2);
    }

    public void use()
    {
        invokeRemote();
    }
}
