package com.microfocus.book;

import javax.servlet.http.HttpSession;

import com.microfocus.cobol.program.IObjectControl;
import com.microfocus.cobol.program.ScaledInteger;
import com.microfocus.cobol.runtimeservices.IRunUnit;
import com.microfocus.cobol.runtimeservices.servlet.ServletRunUnitManager;

public class BookInterface
{
  private static final String READ_RECORD = "1";
  private static final String ADD_RECORD = "2";
  private static final String DELETE_RECORD = "3";
  private static final String NEXT_RECORD = "4";

  private final IRunUnit runUnit;
  private final BookLegacy bookLegacy;
  
  private static final String BOOK_FILE = "C:/work/garethm/JavaWebServicesDemos/COBOLJSPDemo/CobolBook/bookfile.dat";

  public BookInterface(HttpSession session)
  {
    this(ServletRunUnitManager.getManager().GetSessionRunUnit(session));
  }

  public BookInterface(IRunUnit runUnit)
  {
    this.runUnit = runUnit;
    BookLegacy bookLegacy = (BookLegacy) runUnit.GetInstance(BookLegacy.class);

    if (bookLegacy == null)
    {
      bookLegacy = new BookLegacy();
      runUnit.Add(bookLegacy);
    }

    this.bookLegacy = bookLegacy;
  }

  public BookBean readBook(String stockNo) throws JavaBookException
  {
    setFileName(BOOK_FILE);
    Details details = getObject(Details.class);
    FileStatus status = getObject(FileStatus.class);

    details.setStockno(stockNo);
    bookLegacy.BookLegacy(READ_RECORD, details, status);

    throwExceptionIfError(status);

    return bookBeanFromDetails(details);
  }

  public BookBean addBook(BookBean book) throws JavaBookException
  {
    setFileName(BOOK_FILE);
    Details details = getObject(Details.class);
    FileStatus status = getObject(FileStatus.class);

    bookBeanToDetails(book, details);
    bookLegacy.BookLegacy(ADD_RECORD, details, status);

    throwExceptionIfError(status);

    return bookBeanFromDetails(details);
  }

  public BookBean deleteBook(String stockNo) throws JavaBookException
  {
    setFileName(BOOK_FILE);
    Details details = getObject(Details.class);
    FileStatus status = getObject(FileStatus.class);

    details.setStockno(stockNo);
    bookLegacy.BookLegacy(DELETE_RECORD, details, status);

    throwExceptionIfError(status);

    return bookBeanFromDetails(details);
  }

  public BookBean nextBook(String stockNo) throws JavaBookException
  {
    setFileName(BOOK_FILE);
    Details details = getObject(Details.class);
    FileStatus status = getObject(FileStatus.class);

    details.setStockno(stockNo);
    bookLegacy.BookLegacy(NEXT_RECORD, details, status);

    throwExceptionIfError(status);

    return bookBeanFromDetails(details);
  }

  private void setFileName(String filename)
  {
    bookLegacy.SET_FILENAME(getFilenameObject(filename));
  }
  
  private static void bookBeanToDetails(BookBean bean, Details details)
  {
    details.setStockno(bean.getStockno());
    details.setIsbn(Long.parseLong(bean.getIsbn()));
    details.setTitle(bean.getTitle());
    details.setAuthor(bean.getAuthor());
    details.setType(bean.getType());
    details.setRetail(ScaledInteger.parseScaledInteger(bean.getPrice()));

    int onHandInt = Integer.parseInt(bean.getOnhand());
    if (onHandInt < 0)
      throw new RuntimeException(
          "The number of books on hand must be 0 or positive");
    details.setOnhand(onHandInt);

    int soldInt = Integer.parseInt(bean.getSold());
    if (soldInt < 0)
      throw new RuntimeException(
          "The number of books sold must be 0 or positive");
    details.setSold(soldInt);
  }

  private static BookBean bookBeanFromDetails(Details details)
  {
    String stockno = details.getStockno().trim();
    String isbn = "" + details.getIsbn();
    String title = details.getTitle().trim();
    String author = details.getAuthor().trim();
    String type = details.getType().trim();
    String price = details.getRetail().toString();
    String onhand = "" + details.getOnhand();
    String sold = "" + details.getSold();
    ScaledInteger stockvalInt = details.getRetail().multiply(
        new ScaledInteger(details.getOnhand(), 0));
    String stockval = stockvalInt.toString();

    return new BookBean(stockno, isbn, title, author, type, price, onhand,
        sold, stockval);
  }

  private static void throwExceptionIfError(FileStatus statusCode)
      throws JavaBookException
  {
    throwExceptionIfError(statusCode.getFileStatus().trim());
  }

  private static void throwExceptionIfError(String statusCode)
      throws JavaBookException
  {
    if (!"00".equals(statusCode) && !"02".equals(statusCode))
    {
      throw new JavaBookException(statusCode);
    }
  }

  private Filename getFilenameObject(String filename)
  {
    Filename output = getObject(Filename.class);

    output.setFilename(filename);

    return output;
  }

  private <T extends IObjectControl> T getObject(Class<T> cls)
  {
    try
    {
      T output = cls.newInstance();
      runUnit.Add(output);

      return output;
    } catch (Throwable t)
    {
      throw new RuntimeException(t);
    }
  }
}
