package com.vrs.dataservices.dataaccessors;

import java.util.Date;
import java.util.logging.Logger;

import com.softwareag.entirex.aci.BrokerException;
import com.softwareag.entirex.xml.rt.XMLException;
import com.softwareag.entirex.xml.rt.XMLRPCService;
import com.vrs.dataservices.common.XMLRPCServicePool;
import com.vrs.dataservices.exceptions.DataServiceException;

/**
 * This class is a base class that should be used by all classes that
 * will make XML/RPC calls to RIMS via EntireX Broker.
 * <br/><br/>
 * <style type="text/css">
 *  table { border:1px inset gray; }
 *  th { border:1px solid gray; padding-left:6px; padding-right:6px; }
 *  td { border:1px solid gray; padding-left:6px; padding-right:6px; }
 * </style>
 * <table>
 *  <tr>
 *   <th align="center">PWR #</th>
 *   <th align="center">Date</th>
 *   <th align="center">Name</th>
 *   <th align="center">Description</th>
 *  </tr>
 *  <tr>
 *   <td align="left">CP-04474</td>
 *   <td align="left">12/01/2004</td>
 *   <td align="left">edyke</td>
 *   <td align="left">Initial creation</td>
 *  </tr>
 * </table>
 */

/*
 * $Header: com/vrs/dataservices/dataaccessors/DataAccessorNatRPC.java 1.3 2005/09/19 14:51:38EDT Brent Purcell Exp  $
 *
 * $Log: com/vrs/dataservices/dataaccessors/DataAccessorNatRPC.java  $
 * Revision 1.3 2005/09/19 14:51:38EDT Brent Purcell 
 * Fixed javadoc comments
 * Revision 1.2 2005/07/11 15:42:05EDT Brent Purcell 
 * Added/Udpated standard headers...
 *
 */

public class DataAccessorNatRPC {

    public static final String XML_PREAMBLE = "<?xml version='1.0'?>";

    protected String xmlResponse = null;
    protected String brokerId = null;
    protected String brokerService = null;
    protected String brokerUserId = null;
    protected String brokerPassword = null;
    protected String naturalLibrary = null;
    protected String xmmFile = null;
    protected String loggerName = null;

    /**
     * Parameterless constructor
     */
    public DataAccessorNatRPC() {
        super();
    }

    /**
     * Sets the Broker Id to be used when communicating with EntireX Broker
     * @param brokerId
     */
    public void setBrokerId(String brokerId) {
        this.brokerId = brokerId;
    }

    /**
     * Sets the Broker Service id to be used when communicating with EntireX Broker
     * @param brokerService
     */
    public void setBrokerService(String brokerService) {
        this.brokerService = brokerService;
    }

    /**
     * Sets the User id to be used when communicating with EntireX Broker 
     * @param brokerUserId Must be a VSR*** mainframe user id
     */
    public void setBrokerUserId(String brokerUserId) {
        this.brokerUserId = brokerUserId;
    }

    /**
     * Sets the Password to be used when communicating with EntireX Broker
     * @param brokerPassword
     */
    public void setBrokerPassword(String brokerPassword) {
        this.brokerPassword = brokerPassword;
    }

    /**
     * Sets a the name of the logger to be used when logging debug and error information
     * @param loggerName
     */
    public void setLoggerName(String loggerName) {
        this.loggerName = loggerName;
    }

    /**
     * Sets the name of the EntireX Broker IDL/XML mapping file.
     * This method must be called by each class that implements this class.
     * @param xmmFile
     */
    public void setXmmFile(String xmmFile) {
        this.xmmFile = xmmFile;
    }

    /**
     * Sets the name of the Natural Library to be used
     * @param naturalLibrary a valid Natural Library name
     */
    public void setNaturalLibrary(String naturalLibrary) {
        this.naturalLibrary = naturalLibrary;
    }

    /**
     * Gets the current Broker Id
     * @return a String
     */
    public String getBrokerId() {
        return brokerId;
    }

    /**
     * Gets the current Broker service id
     * @return a String
     */
    public String getBrokerService() {
        return brokerService;
    }

    /**
     * Gets the current Broker user id
     * @return a String
     */
    public String getBrokerUserId() {
        return brokerUserId;
    }

    /**
     * Gets the current Broker password
     * @return a String
     */
    public String getBrokerPassword() {
        return brokerPassword;
    }

    /**
     * Gets the name of the current logger
     * @return a String
     */
    public String getLoggerName() {
        return loggerName;
    }

    /**
     * Gets the name of the current IDL/XML mapping file
     * @return a String
     */
    public String getXmmFile() {
        return xmmFile;
    }

    /**
     * Gets the name of the current Natural library
     * @return a String
     */
    public String getNaturalLibrary() {
        return naturalLibrary;
    }

    /**
     * Makes the call to EntireX Broker using XML/RPC
     * @param requestXML and XML document containing a request
     * @throws DataServiceException
     */
    protected void getDataFromRIMS(StringBuffer requestXML) throws DataServiceException {
        Logger logger = null;
        if (loggerName != null) {
            logger = Logger.getLogger(loggerName);
        }
        if (logger != null) {
            logger.fine("sending: " + requestXML.toString());
        }
        Date dstart = new Date();
        /*
         * See if the IDL/XML mapping file has been cached. We cache these
         * because there is a great deal of overhead involved in parsing these
         */
        XMLRPCService service = XMLRPCServicePool.getInstance().aquireService(brokerId,
                brokerService, brokerUserId, brokerPassword, xmmFile, naturalLibrary);
        if (service == null) {
            throw new DataServiceException("200");
        }
        /*
         * Invoke the EntireX Broker service 
         */
        try {
            requestXML.insert(0, XML_PREAMBLE);
            xmlResponse = service.invokeXML(requestXML.toString());
        }
        /*
         * Was there a Broker error? Could the EntireX Broker RPC service be down? 
         */
        catch (BrokerException e) {
            throw new DataServiceException("201", e);
        }
        /*
         * Was there a problem with the XML request?
         */
        catch (XMLException e) {
            throw new DataServiceException("202", e);
        }
        /*
         * Was there some other type of problem?
         */
        catch (Exception e) {
            throw new DataServiceException("001", e);
        }
        /*
         * If a logger has been specified, log the transaction time to it
         */
        if (logger != null) {
            logger.fine("received: " + xmlResponse);
            Date dend = new Date();
            logger
                    .fine("transaction took "
                            + (dend.getTime() - dstart.getTime() + " milliseconds"));
        }
    }

    /**
     * Gets the complete XML document returned by the last EntireX Broker service call
     * @return
     */
    public String getXML() {
        return xmlResponse;
    }
}
