/*
 -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

 Legal notice:

 This free of charge, unsupported Software is provided on an "AS IS" basis,
 without warranty of any kind. The entire risk as to the quality and performance
 of the software is borne by you. Should the software prove defective, you and
 only you assume the entire cost of any service and repair.

 WEBMETHODS DOES NOT MAKE ANY EXPRESS OR IMPLIED WARRANTIES OF ANY KIND
 INCLUDING, BUT NOT LIMITED TO, WARRANTIES OF TITLE, MERCHANTABILITY OR
 FITNESS FOR A PARTICULAR PURPOSE, OR WARRANTIES ARISING BY STATUTE, COURSE
 OF DEALING, CUSTOM OR USAGE, TRADE PRACTICE OR OTHERWISE.

 IN NO EVENT WILL WEBMETHODS BE LIABLE FOR ANY LOST REVENUES, PROFITS,
 BUSINESS OR INTERRUPTION OF BUSINESS OR OTHER SPECIAL, INDIRECT, INCIDENTAL
 OR CONSEQUENTIAL DAMAGES, EVEN IF WEBMETHODS HAS BEEN ADVISED OF THE
 POSSIBILITY OF SUCH DAMAGES, AND NOTWITHSTANDING ANY FAILURE OF ESSENTIAL
 PURPOSE OF ANY LIMITED REMEDY OF ANY KIND, HOWEVER CAUSED AND UNDER ANY
 THEORY OF LIABILITY.

 -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
 */

package com.webmethods.portal.portlet.wp_custauth;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.webmethods.portal.PortalException;
import com.webmethods.portal.bizPolicy.IContext;
import com.webmethods.portal.bizPolicy.impl.ContextFactory;
import com.webmethods.portal.framework.FrameworkException;
import com.webmethods.portal.framework.IAuthManager;
import com.webmethods.portal.framework.IPortalServlet;
import com.webmethods.portal.framework.IWebApplication;
import com.webmethods.portal.framework.auth.IAuthHandler;
import com.webmethods.portal.framework.dispatch.IDispatchHandler;
import com.webmethods.portal.framework.presentation.PresentationData;
import com.webmethods.portal.mech.IMechanics;
import com.webmethods.portal.mech.IMechanicsManager;
import com.webmethods.portal.mech.IMechanicsNames;
import com.webmethods.portal.mech.access.IAuthChallengeResponse;
import com.webmethods.portal.mech.access.IAuthScheme;
import com.webmethods.portal.mech.rules.IRuleMechanics;
import com.webmethods.portal.service.meta2.thing.IThingID;
import com.webmethods.portal.system.IRule;
import com.webmethods.portal.system.IURI;
import com.webmethods.portal.system.PortalSystem;
import com.webmethods.rtl.util.ObjectUtil;
import com.webmethods.rtl.util.StringTools;

/**
 */
public class CustAuthChallengeResponse implements IAuthChallengeResponse {

    protected IURI m_uri;
    protected String m_type;
    protected String m_layout;
    
    public static final String LOGIN_RULE_LAYOUT = "loginPage";
    public static final String LOGIN_PAGE_RULE_FOLDER = "loginpage.rule.folder";
    public static final String RELATIVE_LOGOUT_URL = "?method=logout";

    /**
     * Initialize this authChallengeResponse with it's IAuthScheme
     *
     * @param authScheme
     * @throws com.webmethods.portal.PortalException
     *
     */
    public void init(IAuthScheme authScheme) throws PortalException {
    	
        String redirectURI = authScheme.getRedirectURI();
        m_uri = PortalSystem.getPortalSystem().acquireURI(redirectURI);
        IThingID thingID = (IThingID) m_uri;
        m_type = thingID.getMetaContext().getXTypeService().getXTypeName(thingID.getXTypeID());
        
        //try to figure out if this is a portlet or a portal page
        String typeName = thingID.getMetaContext().getTypeService().getTypeName(thingID.getTypeID());
        m_layout = typeName.toLowerCase();
    }

    /**
     * Handles the request.  The response value tells the {@link com.webmethods.portal.framework.dispatch.DispatchManager}
     * how to further handle the request.  Prior to the {@link com.webmethods.portal.framework.dispatch.DispatchManager} delegating to the IDispatchHandler,
     * a PresentationData is associated with the HttpServletRequest. The following response are valid:<br>
     * <UL>
     * <LI>DISP_RESPONSE_NONE:  The work is complete, don't do anything further
     * <LI>DISP_RESPONSE_REDIRECT:  Redirect this request with an http Redirect using the {@link com.webmethods.portal.framework.presentation.PresentationData}
     * <LI>DISP_RESPONSE_FORWARD: Forward the request to another IDispatchHandler
     * <LI>DISP_RESPONSE_FORWARD_RAW: Forward the request to another IDispatchHandler without checking the type
     * <LI>DISP_RESPONSE_FORWARD_DEFAULT_GET: Forward the request to the default get IDispatchHandler for the requested resource
     * <LI>DISP_RESPONSE_STREAM: Stream the {@link com.webmethods.portal.framework.presentation.PresentationData} back to the user
     * <LI>DISP_RESPONSE_PRES: Delegate to the {@link com.webmethods.portal.framework.IPresentationManager}
     * </UL>
     *
     * @param request  the current HttpServletRequest
     * @param response the current HttpServletResponse
     */
    public int handle(HttpServletRequest request, HttpServletResponse response) throws FrameworkException {
        try {
            //check rules first

            IURI loginPageID = null;
            String type = null;
            String layout = null;
            IContext context = ContextFactory.acquireContext(request);
            IMechanicsManager mechanicsManager = (IMechanicsManager) PortalSystem.getMechanicsProvider();
            IMechanics m = mechanicsManager.getMechanics(IMechanicsNames.RULES);
            if (m instanceof IRuleMechanics) {
				IRuleMechanics rulesMech = (IRuleMechanics)m;
				IThingID ruleID = rulesMech.getRuleForUser((IThingID) PortalSystem.getPortalSystem().acquireURI(LOGIN_PAGE_RULE_FOLDER), context.getUserID());
				if (ruleID != null) {
					IRule rule = (IRule) ruleID.getThing();
					if (!StringTools.isEmpty(rule.getTargetURI())) {
						loginPageID = (IThingID) PortalSystem.getPortalSystem().acquireURI(rule.getTargetURI());
					}
				}
			}

            if (loginPageID == null) {
                //set defaults
                loginPageID = m_uri;
                type = m_type;
                layout = m_layout;
            } else {
                if (loginPageID instanceof IThingID) {
                    IThingID thingID = (IThingID)loginPageID;
                    type = thingID.getMetaContext().getXTypeService().getXTypeName(thingID.getXTypeID());
                    layout = LOGIN_RULE_LAYOUT;
                }
            }

            IWebApplication webApplication = (IWebApplication)PortalSystem.getWebAppProvider();
            IPortalServlet portalServlet = webApplication.getPortalServlet();
            portalServlet.setCurrentResource(loginPageID);
            PresentationData.getPresentationData(request).setLayout(layout);
            context.setCurrentResource(loginPageID);
            context.setCurrentResourceType(type);
            return IDispatchHandler.DISP_RESPONSE_PRES;
        } catch (PortalException e) {
            throw new FrameworkException(e);
        }
    }

    /**
     * Returns full URL for logout request.
     * If there is no URL that that a user can dereference to logout,
     * this method returns null.
     * If the URL is a portal URL, it should include the portal servlet path.
     * For example, the full URL to logout of forms auth is
     * '[full-servlet-path]?method=logout'.
     *
     * @param request HTTP request initialized with portal context info.
     * @return Full logout URL or null.
     */
    public String getLogoutURL(HttpServletRequest request) throws PortalException {
        PresentationData pdo = PresentationData.getPresentationData(request);
        return pdo.getFullServletPath() + RELATIVE_LOGOUT_URL;
    }
}
